<?php
/**
 * Script para eliminar el proyecto "Tareas Eventuales" del usuario lizetm
 * Usuario: Lizet Martija (lizetm, auxiliar.rh@rinorisk.com)
 * Proyecto: Tareas Eventuales
 */

require_once __DIR__ . '/config/database.php';

try {
    $db = Database::getConnection();
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "=== ELIMINAR PROYECTO 'TAREAS EVENTUALES' DEL USUARIO LIZETM ===\n\n";
    
    // Paso 1: Encontrar el usuario
    $stmt = $db->prepare("
        SELECT user_id, username, email, full_name 
        FROM Users 
        WHERE email = 'auxiliar.rh@rinorisk.com' 
           OR username = 'lizetm' 
           OR full_name LIKE '%Lizet%Martija%'
    ");
    $stmt->execute();
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        echo "❌ ERROR: No se encontró el usuario Lizet Martija\n";
        exit(1);
    }
    
    $userId = $user['user_id'];
    echo "✅ Usuario encontrado:\n";
    echo "   - ID: {$userId}\n";
    echo "   - Nombre: {$user['full_name']}\n";
    echo "   - Email: {$user['email']}\n";
    echo "   - Username: {$user['username']}\n\n";
    
    // Paso 2: Encontrar el proyecto "Tareas Eventuales"
    $stmt = $db->prepare("
        SELECT project_id, project_name, clan_id, created_by_user_id
        FROM Projects
        WHERE project_name = 'Tareas Eventuales'
    ");
    $stmt->execute();
    $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($projects)) {
        echo "❌ ERROR: No se encontró el proyecto 'Tareas Eventuales'\n";
        exit(1);
    }
    
    echo "📋 Proyectos 'Tareas Eventuales' encontrados: " . count($projects) . "\n";
    foreach ($projects as $proj) {
        echo "   - Proyecto ID: {$proj['project_id']}, Clan ID: {$proj['clan_id']}, Creado por: {$proj['created_by_user_id']}\n";
    }
    echo "\n";
    
    // Paso 3: Verificar relaciones en Project_Participants
    echo "1. VERIFICANDO Project_Participants:\n";
    $projectIds = array_column($projects, 'project_id');
    $placeholders = implode(',', array_fill(0, count($projectIds), '?'));
    
    $stmt = $db->prepare("
        SELECT pp.project_id, pp.user_id, p.project_name, p.clan_id
        FROM Project_Participants pp
        INNER JOIN Projects p ON pp.project_id = p.project_id
        WHERE pp.user_id = ? AND pp.project_id IN ($placeholders)
    ");
    $params = array_merge([$userId], $projectIds);
    $stmt->execute($params);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($participants) > 0) {
        echo "   📋 Encontradas " . count($participants) . " relación(es) en Project_Participants:\n";
        foreach ($participants as $p) {
            echo "      - Proyecto ID: {$p['project_id']} ({$p['project_name']}), Clan: {$p['clan_id']}\n";
        }
    } else {
        echo "   ✅ No hay relaciones en Project_Participants\n";
    }
    
    // Paso 4: Verificar asignaciones de tareas
    echo "\n2. VERIFICANDO asignaciones de tareas:\n";
    $stmt = $db->prepare("
        SELECT COUNT(*) as total
        FROM Task_Assignments ta
        INNER JOIN Tasks t ON ta.task_id = t.task_id
        WHERE t.project_id IN ($placeholders) AND ta.user_id = ?
    ");
    $stmt->execute(array_merge($projectIds, [$userId]));
    $taskAssignments = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $stmt2 = $db->prepare("
        SELECT COUNT(*) as total
        FROM Tasks
        WHERE project_id IN ($placeholders) AND assigned_to_user_id = ?
    ");
    $stmt2->execute(array_merge($projectIds, [$userId]));
    $directAssignments = $stmt2->fetch(PDO::FETCH_ASSOC);
    
    $totalTaskAssignments = $taskAssignments['total'] + $directAssignments['total'];
    echo "   📋 Asignaciones de tareas encontradas: {$totalTaskAssignments}\n";
    if ($taskAssignments['total'] > 0) {
        echo "      - En Task_Assignments: {$taskAssignments['total']}\n";
    }
    if ($directAssignments['total'] > 0) {
        echo "      - Asignaciones directas: {$directAssignments['total']}\n";
    }
    
    $totalRelations = count($participants) + $totalTaskAssignments;
    echo "\n📊 RESUMEN TOTAL: {$totalRelations} relación(es) encontrada(s)\n\n";
    
    if ($totalRelations == 0) {
        echo "✅ El usuario no tiene relación con el proyecto 'Tareas Eventuales'\n";
        exit(0);
    }
    
    // Paso 5: Eliminar relaciones
    echo "🗑️  ELIMINANDO RELACIONES...\n\n";
    
    $db->beginTransaction();
    
    try {
        $deletedCount = 0;
        
        // Eliminar de Project_Participants
        if (count($participants) > 0) {
            $stmt = $db->prepare("
                DELETE pp FROM Project_Participants pp
                WHERE pp.user_id = ? AND pp.project_id IN ($placeholders)
            ");
            $stmt->execute($params);
            $deletedCount = $stmt->rowCount();
            echo "✅ Eliminadas {$deletedCount} relación(es) de Project_Participants\n";
        }
        
        // Eliminar asignaciones de tareas en Task_Assignments
        if ($taskAssignments['total'] > 0) {
            $stmt = $db->prepare("
                DELETE ta FROM Task_Assignments ta
                INNER JOIN Tasks t ON ta.task_id = t.task_id
                WHERE t.project_id IN ($placeholders) AND ta.user_id = ?
            ");
            $stmt->execute(array_merge($projectIds, [$userId]));
            $deleted = $stmt->rowCount();
            echo "✅ Eliminadas {$deleted} asignación(es) de Task_Assignments\n";
            $deletedCount += $deleted;
        }
        
        // Eliminar asignaciones directas en Tasks
        if ($directAssignments['total'] > 0) {
            $stmt = $db->prepare("
                UPDATE Tasks 
                SET assigned_to_user_id = NULL 
                WHERE project_id IN ($placeholders) AND assigned_to_user_id = ?
            ");
            $stmt->execute(array_merge($projectIds, [$userId]));
            $deleted = $stmt->rowCount();
            echo "✅ Eliminadas {$deleted} asignación(es) directa(s)\n";
            $deletedCount += $deleted;
        }
        
        $db->commit();
        
        echo "\n✅ ÉXITO: Se eliminaron {$deletedCount} relación(es) en total\n";
        echo "✅ Usuario {$user['full_name']} ({$user['email']}) ya no tiene relación con el proyecto 'Tareas Eventuales'\n";
        
        // Verificación final
        echo "\n🔍 VERIFICACIÓN FINAL:\n";
        
        $stmt = $db->prepare("
            SELECT COUNT(*) as remaining
            FROM Project_Participants pp
            WHERE pp.user_id = ? AND pp.project_id IN ($placeholders)
        ");
        $stmt->execute($params);
        $remaining = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stmt2 = $db->prepare("
            SELECT COUNT(*) as remaining
            FROM Task_Assignments ta
            INNER JOIN Tasks t ON ta.task_id = t.task_id
            WHERE t.project_id IN ($placeholders) AND ta.user_id = ?
        ");
        $stmt2->execute(array_merge($projectIds, [$userId]));
        $remainingTasks = $stmt2->fetch(PDO::FETCH_ASSOC);
        
        $stmt3 = $db->prepare("
            SELECT COUNT(*) as remaining_direct
            FROM Tasks
            WHERE project_id IN ($placeholders) AND assigned_to_user_id = ?
        ");
        $stmt3->execute(array_merge($projectIds, [$userId]));
        $remainingDirect = $stmt3->fetch(PDO::FETCH_ASSOC);
        
        $totalRemaining = $remaining['remaining'] + $remainingTasks['remaining'] + $remainingDirect['remaining_direct'];
        
        if ($totalRemaining == 0) {
            echo "✅ Verificación: No quedan relaciones (eliminación completa)\n";
        } else {
            echo "⚠️  ADVERTENCIA: Aún quedan {$totalRemaining} relación(es)\n";
        }
        
    } catch (Exception $e) {
        $db->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    echo "\n❌ ERROR: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
    exit(1);
}

