<?php
/**
 * Script para ejecutar migración de múltiples clanes
 * Ejecutar una sola vez: http://localhost/run_migration.php
 */

require_once __DIR__ . '/../config/database.php';

try {
    $db = Database::getConnection();
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "<h2>Iniciando migración para soporte de múltiples clanes...</h2>\n";
    
    // 1. Crear tabla User_Active_Clan
    echo "<p>1. Creando tabla User_Active_Clan...</p>\n";
    $sql = "
        CREATE TABLE IF NOT EXISTS User_Active_Clan (
            user_id INT NOT NULL,
            active_clan_id INT NOT NULL,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (user_id),
            FOREIGN KEY (user_id) REFERENCES Users(user_id) ON DELETE CASCADE,
            FOREIGN KEY (active_clan_id) REFERENCES Clans(clan_id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ";
    $db->exec($sql);
    echo "<p style='color: green;'>✓ Tabla User_Active_Clan creada exitosamente</p>\n";
    
    // 2. Inicializar clan activo para usuarios existentes
    echo "<p>2. Inicializando clan activo para usuarios existentes...</p>\n";
    $sql = "
        INSERT INTO User_Active_Clan (user_id, active_clan_id)
        SELECT cm.user_id, MIN(cm.clan_id)
        FROM Clan_Members cm
        WHERE NOT EXISTS (
            SELECT 1 FROM User_Active_Clan uac WHERE uac.user_id = cm.user_id
        )
        GROUP BY cm.user_id
        ON DUPLICATE KEY UPDATE active_clan_id = active_clan_id
    ";
    $stmt = $db->exec($sql);
    echo "<p style='color: green;'>✓ Clan activo inicializado para usuarios existentes</p>\n";
    
    // 3. Crear índices
    echo "<p>3. Creando índices para mejorar rendimiento...</p>\n";
    try {
        $db->exec("CREATE INDEX idx_clan_members_user ON Clan_Members(user_id)");
        echo "<p style='color: green;'>✓ Índice idx_clan_members_user creado</p>\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
            echo "<p style='color: orange;'>- Índice idx_clan_members_user ya existe</p>\n";
        } else {
            throw $e;
        }
    }
    
    try {
        $db->exec("CREATE INDEX idx_clan_members_clan ON Clan_Members(clan_id)");
        echo "<p style='color: green;'>✓ Índice idx_clan_members_clan creado</p>\n";
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
            echo "<p style='color: orange;'>- Índice idx_clan_members_clan ya existe</p>\n";
        } else {
            throw $e;
        }
    }
    
    // 4. Crear vista
    echo "<p>4. Creando vista User_Clans_View...</p>\n";
    $sql = "
        CREATE OR REPLACE VIEW User_Clans_View AS
        SELECT 
            u.user_id,
            u.username,
            u.full_name,
            u.email,
            GROUP_CONCAT(DISTINCT c.clan_id ORDER BY c.clan_name SEPARATOR ',') as clan_ids,
            GROUP_CONCAT(DISTINCT c.clan_name ORDER BY c.clan_name SEPARATOR ', ') as clan_names,
            uac.active_clan_id,
            ac.clan_name as active_clan_name
        FROM Users u
        LEFT JOIN Clan_Members cm ON u.user_id = cm.user_id
        LEFT JOIN Clans c ON cm.clan_id = c.clan_id
        LEFT JOIN User_Active_Clan uac ON u.user_id = uac.user_id
        LEFT JOIN Clans ac ON uac.active_clan_id = ac.clan_id
        GROUP BY u.user_id, u.username, u.full_name, u.email, uac.active_clan_id, ac.clan_name
    ";
    $db->exec($sql);
    echo "<p style='color: green;'>✓ Vista User_Clans_View creada exitosamente</p>\n";
    
    // 5. Verificar resultados
    echo "<h3>Verificación de resultados:</h3>\n";
    
    $stmt = $db->query("SELECT COUNT(*) as total FROM User_Active_Clan");
    $result = $stmt->fetch();
    echo "<p>Total de usuarios con clan activo: <strong>" . $result['total'] . "</strong></p>\n";
    
    $stmt = $db->query("
        SELECT 
            u.full_name, 
            COUNT(DISTINCT cm.clan_id) as total_clans,
            GROUP_CONCAT(DISTINCT c.clan_name SEPARATOR ', ') as clans
        FROM Users u
        LEFT JOIN Clan_Members cm ON u.user_id = cm.user_id
        LEFT JOIN Clans c ON cm.clan_id = c.clan_id
        GROUP BY u.user_id, u.full_name
        HAVING COUNT(DISTINCT cm.clan_id) > 1
    ");
    $multiClanUsers = $stmt->fetchAll();
    
    if (count($multiClanUsers) > 0) {
        echo "<p>Usuarios con múltiples clanes:</p>\n";
        echo "<ul>\n";
        foreach ($multiClanUsers as $user) {
            echo "<li><strong>" . htmlspecialchars($user['full_name']) . "</strong>: " . 
                 $user['total_clans'] . " clanes (" . htmlspecialchars($user['clans']) . ")</li>\n";
        }
        echo "</ul>\n";
    } else {
        echo "<p>Aún no hay usuarios con múltiples clanes.</p>\n";
    }
    
    echo "<h2 style='color: green;'>✓ ¡Migración completada exitosamente!</h2>\n";
    echo "<p><a href='/'>Volver al inicio</a></p>\n";
    
} catch (PDOException $e) {
    echo "<h2 style='color: red;'>✗ Error en la migración</h2>\n";
    echo "<p style='color: red;'>" . htmlspecialchars($e->getMessage()) . "</p>\n";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>\n";
}
?>
