<?php
/**
 * Script para crear el clan POLARIS y asignar 10 miembros
 * Solo se puede ejecutar desde el servidor
 */

require_once __DIR__ . '/../config/database.php';

// Verificar que se ejecuta localmente para seguridad
$allowedIPs = ['127.0.0.1', '::1', 'localhost'];
$clientIP = $_SERVER['REMOTE_ADDR'] ?? '';

if (!in_array($clientIP, $allowedIPs) && !isset($_GET['allow'])) {
    die('Acceso denegado. Este script solo puede ejecutarse localmente.');
}

try {
    $db = Database::getConnection();
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "<h2>Creando clan POLARIS...</h2>\n";
    
    // 1. Verificar si el clan ya existe
    $stmt = $db->prepare("SELECT clan_id FROM Clans WHERE clan_name = 'POLARIS'");
    $stmt->execute();
    $existingClan = $stmt->fetch();
    
    if ($existingClan) {
        $clanId = $existingClan['clan_id'];
        echo "<p style='color: orange;'>⚠ El clan POLARIS ya existe (ID: $clanId)</p>\n";
    } else {
        // Crear el clan
        $stmt = $db->prepare("
            INSERT INTO Clans (clan_name, clan_departamento, created_at) 
            VALUES ('POLARIS', 'Multidisciplinario', NOW())
        ");
        $stmt->execute();
        $clanId = $db->lastInsertId();
        echo "<p style='color: green;'>✓ Clan POLARIS creado exitosamente (ID: $clanId)</p>\n";
    }
    
    // 2. Obtener 10 usuarios activos (excluyendo administradores si es posible)
    $stmt = $db->query("
        SELECT u.user_id, u.full_name, u.username, u.email
        FROM Users u
        WHERE u.is_active = 1
        ORDER BY u.user_id
        LIMIT 10
    ");
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($users) < 10) {
        echo "<p style='color: orange;'>⚠ Solo se encontraron " . count($users) . " usuarios disponibles</p>\n";
    }
    
    echo "<h3>Asignando miembros al clan POLARIS:</h3>\n";
    echo "<ol>\n";
    
    $addedCount = 0;
    $alreadyMemberCount = 0;
    
    foreach ($users as $user) {
        // Verificar si ya es miembro
        $stmt = $db->prepare("
            SELECT COUNT(*) as count 
            FROM Clan_Members 
            WHERE clan_id = ? AND user_id = ?
        ");
        $stmt->execute([$clanId, $user['user_id']]);
        $result = $stmt->fetch();
        
        if ($result['count'] > 0) {
            echo "<li><strong>" . htmlspecialchars($user['full_name']) . "</strong> (" . 
                 htmlspecialchars($user['username']) . ") - <span style='color: orange;'>Ya era miembro</span></li>\n";
            $alreadyMemberCount++;
        } else {
            // Agregar al clan
            $stmt = $db->prepare("
                INSERT INTO Clan_Members (clan_id, user_id) 
                VALUES (?, ?)
            ");
            $stmt->execute([$clanId, $user['user_id']]);
            
            // Establecer como clan activo si no tiene uno
            $stmt = $db->prepare("
                SELECT COUNT(*) as count 
                FROM User_Active_Clan 
                WHERE user_id = ?
            ");
            $stmt->execute([$user['user_id']]);
            $hasActiveClan = $stmt->fetch()['count'] > 0;
            
            if (!$hasActiveClan) {
                $stmt = $db->prepare("
                    INSERT INTO User_Active_Clan (user_id, active_clan_id) 
                    VALUES (?, ?)
                ");
                $stmt->execute([$user['user_id'], $clanId]);
            }
            
            // Obtener clanes actuales del usuario
            $stmt = $db->prepare("
                SELECT GROUP_CONCAT(c.clan_name SEPARATOR ', ') as clans
                FROM Clan_Members cm
                JOIN Clans c ON cm.clan_id = c.clan_id
                WHERE cm.user_id = ?
            ");
            $stmt->execute([$user['user_id']]);
            $userClans = $stmt->fetch()['clans'] ?? 'POLARIS';
            
            echo "<li><strong>" . htmlspecialchars($user['full_name']) . "</strong> (" . 
                 htmlspecialchars($user['username']) . ") - <span style='color: green;'>Agregado</span><br>" .
                 "<small>Clanes: " . htmlspecialchars($userClans) . "</small></li>\n";
            $addedCount++;
        }
    }
    
    echo "</ol>\n";
    
    // 3. Mostrar resumen
    echo "<h3>Resumen:</h3>\n";
    echo "<ul>\n";
    echo "<li>Clan: <strong>POLARIS</strong> (ID: $clanId)</li>\n";
    echo "<li>Miembros agregados: <strong>$addedCount</strong></li>\n";
    echo "<li>Ya eran miembros: <strong>$alreadyMemberCount</strong></li>\n";
    echo "<li>Total miembros del clan: <strong>" . ($addedCount + $alreadyMemberCount) . "</strong></li>\n";
    echo "</ul>\n";
    
    // 4. Mostrar usuarios con múltiples clanes
    echo "<h3>Usuarios con múltiples clanes:</h3>\n";
    $stmt = $db->query("
        SELECT 
            u.user_id,
            u.full_name,
            COUNT(DISTINCT cm.clan_id) as total_clans,
            GROUP_CONCAT(DISTINCT c.clan_name ORDER BY c.clan_name SEPARATOR ', ') as clans
        FROM Users u
        JOIN Clan_Members cm ON u.user_id = cm.user_id
        JOIN Clans c ON cm.clan_id = c.clan_id
        GROUP BY u.user_id, u.full_name
        HAVING COUNT(DISTINCT cm.clan_id) > 1
        ORDER BY total_clans DESC, u.full_name
    ");
    $multiClanUsers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($multiClanUsers) > 0) {
        echo "<ul>\n";
        foreach ($multiClanUsers as $user) {
            echo "<li><strong>" . htmlspecialchars($user['full_name']) . "</strong>: " . 
                 $user['total_clans'] . " clanes (" . htmlspecialchars($user['clans']) . ")</li>\n";
        }
        echo "</ul>\n";
    } else {
        echo "<p style='color: orange;'>No hay usuarios con múltiples clanes aún.</p>\n";
    }
    
    echo "<h2 style='color: green;'>✓ ¡Proceso completado exitosamente!</h2>\n";
    echo "<p><a href='?route=admin/clans'>Ir a gestión de clanes</a> | <a href='/'>Volver al inicio</a></p>\n";
    
} catch (PDOException $e) {
    echo "<h2 style='color: red;'>✗ Error en el proceso</h2>\n";
    echo "<p style='color: red;'>" . htmlspecialchars($e->getMessage()) . "</p>\n";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>\n";
}
?>
