/**
 * Script Unificado para el Sidebar
 * Maneja la funcionalidad de colapsar/expandir y responsive
 * Funciona igual en TODAS las vistas: dashboard, admin, clan_leader, clan_member
 */

(function() {
    'use strict';
    
    // Esperar a que el DOM esté completamente cargado
    function initSidebar() {
        const sidebar = document.getElementById('mainSidebar');
        const hamburgerBtn = document.getElementById('hamburgerBtn');
        const sidebarOverlay = document.getElementById('sidebarOverlay');
        const body = document.body;
        
        if (!sidebar) {
            console.warn('⚠️ Sidebar no encontrado');
            return;
        }
        
        console.log('✅ Sidebar encontrado e inicializado');
        
        // Verificar que el botón hamburguesa existe
        if (!hamburgerBtn) {
            console.error('❌ Botón hamburguesa NO encontrado en el DOM');
        } else {
            console.log('✅ Botón hamburguesa encontrado:', hamburgerBtn);
        }
        
        // Detectar ruta actual
        const currentRoute = new URLSearchParams(window.location.search).get('route');
        console.log('📍 Ruta actual:', currentRoute);
        
        // Función simplificada - solo remueve clase collapsed, sin manipular estilos
        function ensureSidebarVisibility() {
            document.documentElement.classList.remove('sidebar-collapsed');
            sidebar.classList.remove('collapsed');
            body.classList.remove('sidebar-collapsed');
        }
        
        // Función para abrir/cerrar sidebar en móvil
        function toggleMobileSidebar() {
            sidebar.classList.toggle('mobile-open');
            if (sidebarOverlay) {
                sidebarOverlay.classList.toggle('active');
            }
            body.style.overflow = sidebar.classList.contains('mobile-open') ? 'hidden' : '';
        }
        
        // Función para colapsar/expandir sidebar
        function toggleSidebar() {
            console.log('🔄 toggleSidebar llamado');
            
            const wasCollapsed = sidebar.classList.contains('collapsed');
            console.log('📊 Estado actual del sidebar (colapsado):', wasCollapsed);
            
            // Toggle de clases en html, body y sidebar
            document.documentElement.classList.toggle('sidebar-collapsed');
            sidebar.classList.toggle('collapsed');
            body.classList.toggle('sidebar-collapsed');
            
            const isNowCollapsed = sidebar.classList.contains('collapsed');
            console.log('📊 Nuevo estado del sidebar (colapsado):', isNowCollapsed);
            
            // Guardar estado en localStorage
            localStorage.setItem('sidebarCollapsed', isNowCollapsed ? 'true' : 'false');
            console.log('💾 Estado guardado en localStorage:', isNowCollapsed ? 'true' : 'false');
        }
        
        // Restaurar estado del sidebar desde localStorage
        const savedState = localStorage.getItem('sidebarCollapsed');
        
        // Aplicar estado colapsado si fue guardado
        if (savedState === 'true' && window.innerWidth > 1024) {
            // La clase ya fue agregada en el <head>, solo agregar al sidebar y body
            sidebar.classList.add('collapsed');
            body.classList.add('sidebar-collapsed');
            console.log('📦 Sidebar colapsado (estado guardado)');
        } else {
            // Asegurar que no esté colapsado si no hay estado guardado
            document.documentElement.classList.remove('sidebar-collapsed');
        }
        
        // Cerrar sidebar móvil al hacer click en overlay
        if (sidebarOverlay) {
            sidebarOverlay.addEventListener('click', toggleMobileSidebar);
        }
        
        // Función para manejar el click del botón hamburguesa
        function handleHamburgerClick(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            
            console.log('🖱️ Click en botón hamburguesa detectado');
            
            // Detectar si estamos en desktop o móvil
            const isDesktop = window.innerWidth > 1024;
            console.log('📱 Tamaño de ventana:', window.innerWidth, 'Desktop:', isDesktop);
            
            if (isDesktop) {
                // Desktop: colapsar/expandir
                console.log('🖥️ Ejecutando toggleSidebar (desktop)');
                toggleSidebar();
            } else {
                // Móvil: abrir/cerrar overlay
                console.log('📱 Ejecutando toggleMobileSidebar (móvil)');
                toggleMobileSidebar();
            }
            
            return false;
        }
        
        // Toggle desde botón hamburguesa - Usar múltiples métodos para asegurar que funcione
        if (hamburgerBtn) {
            console.log('✅ Botón hamburguesa encontrado, agregando event listeners');
            
            // Remover listeners previos si existen
            hamburgerBtn.removeEventListener('click', handleHamburgerClick);
            
            // Agregar listener principal
            hamburgerBtn.addEventListener('click', handleHamburgerClick, true); // useCapture = true
            
            // También agregar onclick inline como fallback
            hamburgerBtn.onclick = handleHamburgerClick;
            
            // Verificar que el listener se agregó
            console.log('✅ Event listener agregado al botón hamburguesa');
            
            // Hacer el botón visible y clickeable
            hamburgerBtn.style.display = 'flex';
            hamburgerBtn.style.pointerEvents = 'auto';
            hamburgerBtn.style.cursor = 'pointer';
            hamburgerBtn.setAttribute('tabindex', '0');
        } else {
            console.error('❌ Botón hamburguesa NO encontrado!');
            
            // Intentar encontrarlo de nuevo después de un delay
            setTimeout(function() {
                const btn = document.getElementById('hamburgerBtn');
                if (btn) {
                    console.log('✅ Botón hamburguesa encontrado en segundo intento');
                    btn.addEventListener('click', handleHamburgerClick, true);
                    btn.onclick = handleHamburgerClick;
                }
            }, 1000);
        }
        
        // También usar delegación de eventos en el documento como respaldo
        document.addEventListener('click', function(e) {
            if (e.target && (e.target.id === 'hamburgerBtn' || e.target.closest('#hamburgerBtn'))) {
                console.log('🖱️ Click detectado mediante delegación de eventos');
                handleHamburgerClick(e);
            }
        }, true); // useCapture = true para capturar antes que otros listeners
        
        
        // Cerrar sidebar móvil al hacer click en un enlace
        const sidebarLinks = sidebar.querySelectorAll('.sidebar-link');
        sidebarLinks.forEach(link => {
            link.addEventListener('click', function() {
                if (window.innerWidth <= 1024) {
                    toggleMobileSidebar();
                }
            });
        });
        
        // Manejar cambios de tamaño de ventana
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            if (window.innerWidth > 1024) {
                // Desktop: remover clases móviles
                sidebar.classList.remove('mobile-open');
                if (sidebarOverlay) sidebarOverlay.classList.remove('active');
                body.style.overflow = '';
            } else {
                // Mobile: remover clases de colapso
                sidebar.classList.remove('collapsed');
                body.classList.remove('sidebar-collapsed');
            }
        }, 250);
    });
    
        // Exponer funciones globalmente si es necesario
        window.toggleSidebar = toggleSidebar;
        window.toggleMobileSidebar = toggleMobileSidebar;
        window.ensureSidebarVisibility = ensureSidebarVisibility;
    }
    
    // Inicializar cuando el DOM esté listo
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            console.log('📄 DOM cargado, inicializando sidebar...');
            initSidebar();
        });
    } else {
        // DOM ya está listo
        console.log('📄 DOM ya estaba listo, inicializando sidebar...');
        initSidebar();
    }
    
    // Reintento eliminado para evitar efectos visuales
})();

/**
 * Selector de Clan - Maneja el cambio de clan activo
 */
(function() {
    'use strict';
    
    function initClanSelector() {
        const clanSelector = document.getElementById('clanSelectorDropdown');
        
        if (!clanSelector) {
            console.log('ℹ️ Selector de clan no disponible (usuario con un solo clan)');
            return;
        }
        
        console.log('✅ Selector de clan inicializado');
        
        clanSelector.addEventListener('change', async function() {
            const selectedClanId = this.value;
            const selectorContainer = this.closest('.clan-selector');
            
            if (!selectedClanId) {
                return;
            }
            
            console.log('🔄 Cambiando a clan ID:', selectedClanId);
            
            // Agregar clase de loading
            if (selectorContainer) {
                selectorContainer.classList.add('changing');
            }
            
            try {
                const formData = new FormData();
                formData.append('clan_id', selectedClanId);
                
                const response = await fetch('?route=dashboard/switch-clan', {
                    method: 'POST',
                    body: formData,
                    credentials: 'same-origin'
                });
                
                const data = await response.json();
                
                if (data.success) {
                    console.log('✅ Clan cambiado exitosamente');
                    
                    // Animación de cambio exitoso
                    if (selectorContainer) {
                        selectorContainer.classList.remove('changing');
                        selectorContainer.classList.add('changed');
                        setTimeout(() => {
                            selectorContainer.classList.remove('changed');
                        }, 600);
                    }
                    
                    // SOLUCIÓN: Verificar tab "Equipo" ANTES de recargar
                    if (typeof window.checkTeamTabVisibility === 'function') {
                        window.checkTeamTabVisibility();
                    }
                    
                    // Recargar página después de un breve delay SOLO si es necesario
                    // Para dashboards, NO recargar, solo actualizar el tab
                    const currentRoute = window.location.search;
                    if (!currentRoute.includes('route=clan_leader') && !currentRoute.includes('route=clan_member')) {
                        setTimeout(() => {
                            window.location.reload();
                        }, 300);
                    } else {
                        console.log('📊 Dashboard detectado - actualizando sin recargar');
                    }
                } else {
                    console.error('❌ Error al cambiar clan:', data.message);
                    alert('Error al cambiar de clan: ' + (data.message || 'Error desconocido'));
                    
                    // Restaurar selección anterior
                    if (selectorContainer) {
                        selectorContainer.classList.remove('changing');
                    }
                }
            } catch (error) {
                console.error('❌ Error de conexión:', error);
                alert('Error de conexión al cambiar de clan');
                
                if (selectorContainer) {
                    selectorContainer.classList.remove('changing');
                }
            }
        });
    }
    
    // Inicializar cuando el DOM esté listo
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initClanSelector);
    } else {
        initClanSelector();
    }
})();

