// Función global para mostrar notificaciones toast
window.showToast = function(message, type = 'info') {
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: var(--bg-primary);
        border: 1px solid var(--bg-accent);
        border-radius: var(--radius-md);
        padding: var(--spacing-md) var(--spacing-lg);
        box-shadow: var(--shadow-lg);
        z-index: 10000;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        max-width: 300px;
        color: var(--text-primary);
    `;
    
    if (type === 'success') {
        toast.style.borderLeftColor = 'var(--success)';
        toast.style.borderLeftWidth = '4px';
    } else if (type === 'error') {
        toast.style.borderLeftColor = 'var(--error)';
        toast.style.borderLeftWidth = '4px';
    } else if (type === 'warning') {
        toast.style.borderLeftColor = 'var(--warning)';
        toast.style.borderLeftWidth = '4px';
    } else if (type === 'info') {
        toast.style.borderLeftColor = 'var(--info)';
        toast.style.borderLeftWidth = '4px';
    }
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 10);
    
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
};

// Esperar a que el DOM esté completamente cargado
document.addEventListener('DOMContentLoaded', function() {
    
    // Elementos del DOM
    const loginForm = document.getElementById('loginForm');
    const usernameInput = document.getElementById('username');
    const passwordInput = document.getElementById('password');
    const togglePassword = document.getElementById('togglePassword');
    const loginBtn = document.getElementById('loginBtn');
    const loginMessage = document.getElementById('loginMessage');
    
    // Solo ejecutar código de login si estamos en la página de login
    if (!loginForm) {
        // No estamos en la página de login, salir del script
        return;
    }
    
    // Toggle para mostrar/ocultar contraseña
    if (togglePassword) {
        togglePassword.addEventListener('click', function() {
        const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
        passwordInput.setAttribute('type', type);
        
        // Cambiar el ícono
        this.classList.toggle('fa-eye');
        this.classList.toggle('fa-eye-slash');
        });
    }
    
    // Funciones de validación simplificadas para el nuevo diseño
    
    // Mostrar mensaje de login
    function showLoginMessage(message, type) {
        loginMessage.textContent = message;
        loginMessage.className = `login__message show ${type}`;
    }
    
    // Ocultar mensaje de login
    function hideLoginMessage() {
        loginMessage.classList.remove('show');
    }
    
    // Función para mostrar/ocultar loader del botón
    function toggleButtonLoader(show) {
        if (show) {
            loginBtn.classList.add('processing');
            loginBtn.disabled = true;
        } else {
            loginBtn.classList.remove('processing');
            loginBtn.disabled = false;
        }
    }
    
    // Función para efecto ripple en el botón
    function createRipple(event) {
        const button = event.currentTarget;
        const ripple = document.createElement('span');
        ripple.classList.add('ripple');
        
        const rect = button.getBoundingClientRect();
        const x = event.clientX - rect.left;
        const y = event.clientY - rect.top;
        
        ripple.style.left = x + 'px';
        ripple.style.top = y + 'px';
        
        button.appendChild(ripple);
        
        setTimeout(() => {
            ripple.remove();
        }, 500);
    }
    
    // Agregar efecto ripple al botón
    if (loginBtn) {
        loginBtn.addEventListener('click', createRipple);
    }
    
    // Manejar envío del formulario
    if (loginForm) {
        loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Ocultar mensajes previos
        hideLoginMessage();
        
        // Validar formulario
        const username = usernameInput ? usernameInput.value.trim() : '';
        const password = passwordInput ? passwordInput.value : '';
        
        if (username.length === 0 || password.length === 0) {
            showLoginMessage('Por favor completa todos los campos', 'error');
            return;
        }
        
        if (username.length < 3) {
            showLoginMessage('El usuario debe tener al menos 3 caracteres', 'error');
            return;
        }
        
        if (password.length < 6) {
            showLoginMessage('La contraseña debe tener al menos 6 caracteres', 'error');
            return;
        }
        
        // Mostrar loader
        toggleButtonLoader(true);
        
        // Preparar datos para enviar
        const formData = new FormData();
        formData.append('username', username);
        formData.append('password', password);
        
        // Enviar datos con AJAX (compatible con y sin mod_rewrite)
        fetch('?route=process-login', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loginBtn.classList.remove('processing');
                loginBtn.classList.add('success');
                showLoginMessage(data.message, 'success');
                
                // Mostrar pantalla de carga con animación Lottie por 5 segundos
                setTimeout(() => {
                    showLottieLoader();
                    
                    // Redireccionar después de 5 segundos
                    setTimeout(() => {
                        const redirectRoute = data.redirect || 'clan_member';
                        window.location.href = '?route=' + redirectRoute;
                    }, 5000);
                }, 1000);
                
            } else {
                toggleButtonLoader(false);
                showLoginMessage(data.message, 'error');
            }
        })
        .catch(error => {
            toggleButtonLoader(false);
            console.error('Error:', error);
            showLoginMessage('Error de conexión. Por favor intenta de nuevo.', 'error');
        });
        });
    }
    
    // Animaciones adicionales para el login
    const loginInputs = document.querySelectorAll('.login__input');
    if (loginInputs.length > 0) {
        loginInputs.forEach(input => {
            input.addEventListener('focus', function() {
                this.parentElement.style.borderBottomColor = 'rgba(255, 255, 255, 0.6)';
            });
            
            input.addEventListener('blur', function() {
                this.parentElement.style.borderBottomColor = 'rgba(255, 255, 255, 0.2)';
            });
        });
    }

    // === SISTEMA DE TEMAS ===
    const dashboard = document.querySelector('.modern-dashboard');
    const themeButtons = document.querySelectorAll('.theme-btn');
    
    if (dashboard && themeButtons.length > 0) {
        // Cargar tema guardado
        const savedTheme = localStorage.getItem('rinotrack-theme') || 'default';
        setTheme(savedTheme);
        
        // Event listeners para los botones de tema
        themeButtons.forEach(button => {
            if (button.dataset.theme === savedTheme) {
                button.classList.add('active');
            }
            
            button.addEventListener('click', function() {
                const theme = this.dataset.theme;
                setTheme(theme);
                
                // Actualizar botón activo
                themeButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                // Guardar tema en localStorage
                localStorage.setItem('rinotrack-theme', theme);
                
                // Efecto visual al cambiar tema
                dashboard.style.transform = 'scale(0.99)';
                setTimeout(() => {
                    dashboard.style.transform = 'scale(1)';
                }, 150);
            });
        });
        
        function setTheme(theme) {
            dashboard.setAttribute('data-theme', theme);
            document.body.setAttribute('data-theme', theme);
        }
    }
    
    // === ANIMACIONES DE ENTRADA ===
    const animatedElements = document.querySelectorAll('.animate-fade-in');
    if (animatedElements.length > 0) {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.style.animation = 'fadeIn 0.6s ease-out forwards';
                }
            });
        }, { threshold: 0.1 });
        
        animatedElements.forEach(el => {
            observer.observe(el);
        });
    }
    
    // === EFECTOS HOVER MEJORADOS ===
    const cards = document.querySelectorAll('.stat-card, .content-card');
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-8px) scale(1.02)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
    
    // === TOOLTIPS DINÁMICOS ===
    const tooltipElements = document.querySelectorAll('[title]');
    tooltipElements.forEach(element => {
        element.addEventListener('mouseenter', function(e) {
            const tooltip = document.createElement('div');
            tooltip.textContent = this.getAttribute('title');
            tooltip.className = 'dynamic-tooltip';
            tooltip.style.cssText = `
                position: fixed;
                background: var(--bg-dark);
                color: var(--text-white);
                padding: 8px 12px;
                border-radius: 8px;
                font-size: 0.8rem;
                z-index: 10000;
                pointer-events: none;
                opacity: 0;
                transition: opacity 0.2s ease;
                box-shadow: var(--shadow-lg);
            `;
            
            document.body.appendChild(tooltip);
            
            const rect = tooltip.getBoundingClientRect();
            tooltip.style.left = (e.clientX - rect.width / 2) + 'px';
            tooltip.style.top = (e.clientY - rect.height - 10) + 'px';
            
            setTimeout(() => tooltip.style.opacity = '1', 10);
            
            this.addEventListener('mouseleave', function() {
                tooltip.remove();
            });
        });
    });
    

}); 

/* ===== SISTEMA DE MODALES DE CONFIRMACIÓN ===== */

// Función global para mostrar modales de confirmación personalizados
window.showConfirmationModal = function(options) {
    const {
        title = 'Confirmar Acción',
        message = '¿Estás seguro de que quieres realizar esta acción?',
        icon = 'warning',
        confirmText = 'Confirmar',
        cancelText = 'Cancelar',
        onConfirm = null,
        onCancel = null,
        type = 'warning' // warning, info, success
    } = options;

    // Crear el HTML del modal
    const modalHTML = `
        <div class="confirmation-modal-overlay" id="confirmationModalOverlay">
            <div class="confirmation-modal" id="confirmationModal">
                <div class="confirmation-modal-header">
                    <h3 class="confirmation-modal-title">${title}</h3>
                </div>
                <div class="confirmation-modal-body">
                    <i class="fas fa-${getIconForType(type)} confirmation-modal-icon ${type}"></i>
                    <p class="confirmation-modal-message">${message}</p>
                    <div class="confirmation-modal-actions">
                        <button class="confirmation-modal-btn cancel" id="confirmationCancelBtn">
                            ${cancelText}
                        </button>
                        <button class="confirmation-modal-btn confirm" id="confirmationConfirmBtn">
                            ${confirmText}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Agregar el modal al DOM
    document.body.insertAdjacentHTML('beforeend', modalHTML);

    const overlay = document.getElementById('confirmationModalOverlay');
    const modal = document.getElementById('confirmationModal');
    const confirmBtn = document.getElementById('confirmationConfirmBtn');
    const cancelBtn = document.getElementById('confirmationCancelBtn');

    // Mostrar el modal con animación
    setTimeout(() => {
        overlay.classList.add('show');
    }, 10);

    // Función para cerrar el modal
    const closeModal = (result) => {
        overlay.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(overlay);
            if (result && onConfirm) {
                onConfirm();
            } else if (!result && onCancel) {
                onCancel();
            }
        }, 300);
    };

    // Event listeners
    confirmBtn.addEventListener('click', () => closeModal(true));
    cancelBtn.addEventListener('click', () => closeModal(false));
    
    // Cerrar al hacer clic en el overlay
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
            closeModal(false);
        }
    });

    // Cerrar con Escape
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeModal(false);
            document.removeEventListener('keydown', handleEscape);
        }
    };
    document.addEventListener('keydown', handleEscape);

    // Enfocar el botón de cancelar por defecto
    cancelBtn.focus();
};

// Función para obtener el icono según el tipo
function getIconForType(type) {
    const icons = {
        warning: 'exclamation-triangle',
        info: 'info-circle',
        success: 'check-circle',
        danger: 'times-circle',
        question: 'question-circle'
    };
    return icons[type] || 'exclamation-triangle';
}

// Función de conveniencia para confirmaciones de eliminación
window.confirmDelete = function(message, onConfirm, onCancel = null) {
    showConfirmationModal({
        title: 'Confirmar Eliminación',
        message: message,
        icon: 'warning',
        confirmText: 'Eliminar',
        cancelText: 'Cancelar',
        type: 'warning',
        onConfirm: onConfirm,
        onCancel: onCancel
    });
};

// Función de conveniencia para confirmaciones de información
window.confirmInfo = function(message, onConfirm, onCancel = null) {
    showConfirmationModal({
        title: 'Información',
        message: message,
        icon: 'info',
        confirmText: 'Aceptar',
        cancelText: 'Cancelar',
        type: 'info',
        onConfirm: onConfirm,
        onCancel: onCancel
    });
};

// Función de conveniencia para confirmaciones de éxito
window.confirmSuccess = function(message, onConfirm, onCancel = null) {
    showConfirmationModal({
        title: 'Éxito',
        message: message,
        icon: 'check',
        confirmText: 'Continuar',
        cancelText: 'Cancelar',
        type: 'success',
        onConfirm: onConfirm,
        onCancel: onCancel
    });
};

// === PANTALLA DE CARGA LOTTIE ===
function showLottieLoader() {
    // Crear overlay
    const overlay = document.createElement('div');
    overlay.id = 'lottieLoader';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255, 255, 255, 0.95);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 10000;
        backdrop-filter: blur(5px);
        animation: fadeIn 0.5s ease-in-out;
    `;
    
    // Crear contenedor para la animación
    const container = document.createElement('div');
    container.style.cssText = `
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 20px;
    `;
    
    // Crear elemento dotlottie-wc
    const lottieElement = document.createElement('dotlottie-wc');
    lottieElement.setAttribute('src', 'https://lottie.host/4a3d7c50-8d2d-4b56-bee3-e98ff015eaf0/jFdiNuPSVI.lottie');
    lottieElement.style.cssText = `
        width: 300px;
        height: 300px;
        filter: drop-shadow(0 4px 20px rgba(0, 0, 0, 0.1));
    `;
    lottieElement.setAttribute('autoplay', '');
    lottieElement.setAttribute('loop', '');
    
    // Crear texto de carga
    const loadingText = document.createElement('div');
    loadingText.textContent = 'Cargando...';
    loadingText.style.cssText = `
        color: #374151;
        font-size: 24px;
        font-weight: 300;
        text-align: center;
        font-family: 'Open Sans', sans-serif;
        opacity: 0.9;
        animation: pulse 2s infinite;
    `;
    
    // Agregar estilos de animación
    const style = document.createElement('style');
    style.textContent = `
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        @keyframes pulse {
            0%, 100% { opacity: 0.9; }
            50% { opacity: 0.5; }
        }
    `;
    document.head.appendChild(style);
    
    // Ensamblar elementos
    container.appendChild(lottieElement);
    container.appendChild(loadingText);
    overlay.appendChild(container);
    
    // Agregar al DOM
    document.body.appendChild(overlay);
    
    // Cargar el script de dotlottie-wc si no está cargado
    if (!document.querySelector('script[src*="dotlottie-wc"]')) {
        const script = document.createElement('script');
        script.src = 'https://unpkg.com/@lottiefiles/dotlottie-wc@0.8.5/dist/dotlottie-wc.js';
        script.type = 'module';
        document.head.appendChild(script);
    }
} 