// Clan Leader Dashboard - JavaScript Interactivo

document.addEventListener('DOMContentLoaded', function () {
    initClanLeaderDashboard();
});

function initClanLeaderDashboard() {
    // Inicializar animaciones
    setTimeout(() => {
        animateNumbers();
        animateProgressBars();
    }, 300);

    // Configurar eventos
    setupEventListeners();
}

// Animaciones de números
function animateNumbers() {
    const statNumbers = document.querySelectorAll('.stat-value');

    statNumbers.forEach(element => {
        const finalNumber = parseInt(element.textContent.replace(/,/g, ''));
        if (isNaN(finalNumber)) return;

        const duration = 1500;
        const start = 0;
        const startTime = performance.now();

        function updateNumber(currentTime) {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);

            const easeOutCubic = 1 - Math.pow(1 - progress, 3);
            const currentNumber = Math.floor(easeOutCubic * finalNumber);

            element.textContent = new Intl.NumberFormat().format(currentNumber);

            if (progress < 1) {
                requestAnimationFrame(updateNumber);
            } else {
                element.textContent = new Intl.NumberFormat().format(finalNumber);
            }
        }

        requestAnimationFrame(updateNumber);
    });
}

// Animar progreso de barras
function animateProgressBars() {
    const progressBars = document.querySelectorAll('.progress-fill-small');

    progressBars.forEach((bar, index) => {
        const targetWidth = bar.style.width;
        const currentWidth = parseFloat(targetWidth) || 0;

        // Solo animar si hay progreso (width > 0)
        if (currentWidth > 0) {
            // Establecer width inicial solo si no está ya establecido
            if (!bar.dataset.animated) {
                bar.style.width = '0%';
                bar.dataset.animated = 'true';

                // Animar con delay escalonado para efecto más suave
                setTimeout(() => {
                    bar.style.transition = 'width 0.8s ease-out';
                    bar.style.width = targetWidth;
                }, 100 + (index * 50));
            }
        }
    });
}

// Configurar listeners de eventos
function setupEventListeners() {
    // Listener para tecla F5 personalizada
    document.addEventListener('keydown', function (e) {
        if (e.key === 'F5' || (e.ctrlKey && e.key === 'r')) {
            e.preventDefault();
            refreshDashboard();
        }
    });

    // Listeners para botones de acción
    setupActionButtons();

    // Listener para tooltips
    setupTooltips();
}

function setupActionButtons() {
    // Botones de acción
    const actionButtons = document.querySelectorAll('.action-btn');
    actionButtons.forEach(button => {
        button.addEventListener('click', function () {
            // Efecto de ripple
            createRippleEffect(this);
        });
    });
}

function createRippleEffect(element) {
    const ripple = document.createElement('span');
    const rect = element.getBoundingClientRect();
    const size = Math.max(rect.width, rect.height);
    const x = event.clientX - rect.left - size / 2;
    const y = event.clientY - rect.top - size / 2;

    ripple.style.width = ripple.style.height = size + 'px';
    ripple.style.left = x + 'px';
    ripple.style.top = y + 'px';
    ripple.classList.add('ripple');

    element.appendChild(ripple);

    setTimeout(() => {
        ripple.remove();
    }, 600);
}

function setupTooltips() {
    const elementsWithTooltips = document.querySelectorAll('[title]');

    elementsWithTooltips.forEach(element => {
        let tooltip;

        element.addEventListener('mouseenter', function () {
            const title = this.getAttribute('title');
            if (!title) return;

            // Crear tooltip
            tooltip = document.createElement('div');
            tooltip.className = 'custom-tooltip';
            tooltip.textContent = title;
            document.body.appendChild(tooltip);

            // Posicionar tooltip
            const rect = this.getBoundingClientRect();
            tooltip.style.left = rect.left + rect.width / 2 - tooltip.offsetWidth / 2 + 'px';
            tooltip.style.top = rect.top - tooltip.offsetHeight - 10 + 'px';

            // Remover title para evitar tooltip nativo
            this.setAttribute('data-original-title', title);
            this.removeAttribute('title');

            // Animar entrada
            setTimeout(() => {
                tooltip.style.opacity = '1';
            }, 10);
        });

        element.addEventListener('mouseleave', function () {
            if (tooltip) {
                tooltip.style.opacity = '0';
                setTimeout(() => {
                    if (tooltip.parentNode) {
                        tooltip.parentNode.removeChild(tooltip);
                    }
                }, 200);
            }

            // Restaurar title original
            const originalTitle = this.getAttribute('data-original-title');
            if (originalTitle) {
                this.setAttribute('title', originalTitle);
                this.removeAttribute('data-original-title');
            }
        });
    });
}

// Función para refrescar dashboard
function refreshDashboard() {
    showToast('Actualizando dashboard...', 'info');
    setTimeout(() => {
        location.reload();
    }, 1000);
}

// Funciones de utilidad
function formatNumber(number) {
    if (number >= 1000000) {
        return (number / 1000000).toFixed(1) + 'M';
    } else if (number >= 1000) {
        return (number / 1000).toFixed(1) + 'K';
    }
    return number.toLocaleString();
}

function getProgressColor(percentage) {
    if (percentage >= 80) return 'var(--success)';
    if (percentage >= 60) return 'var(--warning)';
    if (percentage >= 40) return 'var(--info)';
    return 'var(--error)';
}

// Función para mostrar toast
function showToast(message, type = 'info') {
    // Crear toast si no existe
    let toast = document.getElementById('toast');
    if (!toast) {
        toast = document.createElement('div');
        toast.id = 'toast';
        toast.className = 'toast';
        document.body.appendChild(toast);
    }

    // Configurar contenido
    toast.innerHTML = `
        <div class="toast-content ${type}">
            <i class="fas fa-${getToastIcon(type)}"></i>
            <span>${message}</span>
        </div>
    `;

    // Mostrar toast
    toast.classList.add('show');

    // Ocultar después de 3 segundos
    setTimeout(() => {
        toast.classList.remove('show');
    }, 3000);
}

function getToastIcon(type) {
    switch (type) {
        case 'success': return 'check-circle';
        case 'error': return 'exclamation-circle';
        case 'warning': return 'exclamation-triangle';
        default: return 'info-circle';
    }
}

// Funciones específicas para gestión de miembros
function openAddMemberModal() {
    const modal = document.getElementById('addMemberModal');
    if (modal) {
        modal.style.display = 'flex';
        loadAvailableUsers();
    }
}

function closeAddMemberModal() {
    const modal = document.getElementById('addMemberModal');
    if (modal) {
        modal.style.display = 'none';
    }
}

function loadAvailableUsers() {
    fetch('?route=clan_leader/get-available-users', {
        method: 'GET',
        credentials: 'same-origin'
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('userId');
                if (select) {
                    select.innerHTML = '<option value="">Seleccionar usuario...</option>';

                    data.users.forEach(user => {
                        const option = document.createElement('option');
                        option.value = user.user_id;
                        option.textContent = `${user.full_name} (@${user.username})`;
                        select.appendChild(option);
                    });
                }
            } else {
                showToast('Error al cargar usuarios disponibles', 'error');
            }
        })
        .catch(error => {
            console.error('Error al cargar usuarios:', error);
            showToast('Error de conexión al cargar usuarios', 'error');
        });
}

function removeMember(userId, userName) {
    showConfirmationModal({
        title: 'Confirmar Remoción',
        message: `¿Estás seguro de que quieres remover a ${userName} del clan?`,
        type: 'warning',
        confirmText: 'Remover',
        cancelText: 'Cancelar',
        onConfirm: () => {
            const formData = new FormData();
            formData.append('userId', userId);

            fetch('?route=clan_leader/remove-member', {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showToast(data.message, 'success');
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showToast(data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showToast('Error de conexión', 'error');
                });
        }
    });
}

// Manejar envío del formulario de agregar miembro
document.addEventListener('DOMContentLoaded', function () {
    const addMemberForm = document.getElementById('addMemberForm');
    if (addMemberForm) {
        addMemberForm.addEventListener('submit', function (e) {
            e.preventDefault();

            const formData = new FormData(this);

            fetch('?route=clan_leader/add-member', {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showToast(data.message, 'success');
                        closeAddMemberModal();
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showToast(data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showToast('Error de conexión', 'error');
                });
        });
    }

    // Cerrar modal al hacer clic fuera
    const modal = document.getElementById('addMemberModal');
    if (modal) {
        modal.addEventListener('click', function (e) {
            if (e.target === this) {
                closeAddMemberModal();
            }
        });
    }
});

// Funciones para el calendario de tareas
if (typeof window.clanLeaderTasksData === 'undefined') {
    window.clanLeaderTasksData = [];
}
if (typeof window.clanLeaderCurrentDate === 'undefined') {
    window.clanLeaderCurrentDate = new Date();
}

// Generar calendario
function generateCalendar() {
    const year = window.clanLeaderCurrentDate.getFullYear();
    const month = window.clanLeaderCurrentDate.getMonth();

    // Actualizar título del mes
    const monthNames = ['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio',
        'Julio', 'Agosto', 'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'];
    const monthElement = document.getElementById('currentMonth');
    if (monthElement) {
        monthElement.textContent = `${monthNames[month]} ${year}`;
    }

    const firstDay = new Date(year, month, 1);
    const lastDay = new Date(year, month + 1, 0);
    const startDate = new Date(firstDay);
    startDate.setDate(startDate.getDate() - firstDay.getDay());

    const calendarDays = document.getElementById('calendarDays');
    if (!calendarDays) {
        return;
    }

    calendarDays.innerHTML = '';

    const today = new Date();
    today.setHours(0, 0, 0, 0);

    for (let i = 0; i < 42; i++) {
        const currentDay = new Date(startDate);
        currentDay.setDate(startDate.getDate() + i);

        const dayElement = document.createElement('div');
        dayElement.className = 'calendar-day';

        // Verificar si es otro mes
        if (currentDay.getMonth() !== month) {
            dayElement.classList.add('other-month');
        }

        // Verificar si es hoy
        if (currentDay.getTime() === today.getTime()) {
            dayElement.classList.add('today');
        }

        // Verificar si tiene tareas
        const dayTasks = getTasksForDate(currentDay);
        if (dayTasks.length > 0) {
            dayElement.classList.add('has-tasks');
        }

        const dayNumber = document.createElement('div');
        dayNumber.className = 'day-number';
        dayNumber.textContent = currentDay.getDate();

        const dayTasksInfo = document.createElement('div');
        dayTasksInfo.className = 'day-tasks';
        dayTasksInfo.textContent = `${dayTasks.length} tarea${dayTasks.length !== 1 ? 's' : ''}`;

        dayElement.appendChild(dayNumber);
        dayElement.appendChild(dayTasksInfo);

        // Agregar indicadores de estado
        if (dayTasks.length > 0) {
            const statusCounts = {};
            dayTasks.forEach(task => {
                const status = task.task.status;
                statusCounts[status] = (statusCounts[status] || 0) + 1;
            });

            const indicators = document.createElement('div');
            indicators.style.marginTop = '0.25rem';

            Object.keys(statusCounts).forEach(status => {
                const indicator = document.createElement('span');
                indicator.className = `task-indicator ${status}`;
                indicator.title = `${statusCounts[status]} tarea${statusCounts[status] !== 1 ? 's' : ''} ${status}`;
                indicators.appendChild(indicator);
            });

            dayElement.appendChild(indicators);
        }

        // Agregar evento click
        dayElement.addEventListener('click', () => {
            showTasksForDate(currentDay, dayTasks);
        });

        calendarDays.appendChild(dayElement);
    }
}

// Obtener tareas para una fecha específica
function getTasksForDate(date) {
    const dateStr = date.toISOString().split('T')[0];

    const filteredTasks = window.clanLeaderTasksData.filter(taskData => {
        const taskDate = taskData.task.due_date;
        return taskDate === dateStr;
    });

    return filteredTasks;
}

// Mostrar tareas para una fecha específica
function showTasksForDate(date, tasks) {
    const modal = document.getElementById('taskModal');
    const modalTitle = document.getElementById('modalTitle');
    const modalTaskList = document.getElementById('modalTaskList');

    if (!modal || !modalTitle || !modalTaskList) return;

    const dateStr = date.toLocaleDateString('es-ES', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });

    modalTitle.textContent = `Tareas del ${dateStr}`;
    modalTaskList.innerHTML = '';

    if (tasks.length === 0) {
        modalTaskList.innerHTML = '<p>No hay tareas programadas para este día.</p>';
    } else {
        tasks.forEach(taskData => {
            const task = taskData.task;
            const project = taskData.project;
            const assignedUser = taskData.assigned_user;

            const taskElement = document.createElement('div');
            taskElement.className = 'task-item';

            const isOverdue = new Date(task.due_date) < new Date() && task.status !== 'completed';
            const statusClass = isOverdue ? 'overdue' : task.status;

            taskElement.innerHTML = `
                <div class="task-header">
                    <h4 class="task-title">${task.task_name}</h4>
                    <span class="task-status ${statusClass}">
                        ${isOverdue ? 'Vencida' :
                    task.status === 'pending' ? 'Pendiente' :
                        task.status === 'in_progress' ? 'En Progreso' :
                            task.status === 'completed' ? 'Completada' : 'Cancelada'}
                    </span>
                </div>
                <div class="task-details">
                    <div class="task-project">
                        <i class="fas fa-folder"></i> ${project.project_name}
                    </div>
                    ${assignedUser ? `
                        <div class="task-assigned">
                            <i class="fas fa-user"></i> ${assignedUser.full_name}
                        </div>
                    ` : ''}
                    ${task.description ? `
                        <div style="margin-top: 0.5rem;">
                            <i class="fas fa-align-left"></i> ${task.description}
                        </div>
                    ` : ''}
                </div>
            `;

            modalTaskList.appendChild(taskElement);
        });
    }

    // Agregar botón "Agregar tarea" al final del modal
    const addTaskButton = document.createElement('div');
    addTaskButton.className = 'modal-add-task-section';
    addTaskButton.innerHTML = `
        <button class="btn-add-task-modal" onclick="openAddTaskModal('${date.toISOString().split('T')[0]}')">
            <i class="fas fa-plus"></i>
            Agregar tarea
        </button>
    `;

    modalTaskList.appendChild(addTaskButton);

    modal.classList.add('show');
}

// Cerrar modal
function closeTaskModal() {
    const modal = document.getElementById('taskModal');
    if (modal) {
        modal.classList.remove('show');
    }
}

// Navegación del calendario
function previousMonth() {
    window.clanLeaderCurrentDate.setMonth(window.clanLeaderCurrentDate.getMonth() - 1);
    generateCalendar();
}

function nextMonth() {
    window.clanLeaderCurrentDate.setMonth(window.clanLeaderCurrentDate.getMonth() + 1);
    generateCalendar();
}

// Inicializar calendario cuando se carga la página
document.addEventListener('DOMContentLoaded', () => {
    // Cerrar modal al hacer clic fuera
    document.addEventListener('click', (e) => {
        const modal = document.getElementById('taskModal');
        if (modal && e.target === modal) {
            closeTaskModal();
        }
    });

    // Inicializar calendario si estamos en la página de disponibilidad
    if (document.getElementById('calendarDays')) {
        generateCalendar();
        // Restaurar estado del colapso
        setTimeout(() => {
            restoreCalendarState();
        }, 100);
    }
});

// Función para establecer los datos de tareas (llamada desde PHP)
function setTasksData(data) {
    window.clanLeaderTasksData = data;
    if (document.getElementById('calendarDays')) {
        generateCalendar();
    }
}

// Función para colapsar/expandir el calendario
function toggleCalendar() {
    const container = document.getElementById('calendarContainer');
    const toggle = document.getElementById('calendarToggle');
    const icon = document.getElementById('calendarIcon');
    const text = document.getElementById('calendarToggleText');

    if (!container || !toggle || !icon || !text) return;

    const isCollapsed = container.classList.contains('collapsed');

    if (isCollapsed) {
        // Expandir
        container.classList.remove('collapsed');
        toggle.classList.remove('collapsed');
        icon.className = 'fas fa-chevron-up';
        text.textContent = 'Ocultar';
        localStorage.setItem('calendarCollapsed', 'false');
    } else {
        // Colapsar
        container.classList.add('collapsed');
        toggle.classList.add('collapsed');
        icon.className = 'fas fa-chevron-down';
        text.textContent = 'Mostrar';
        localStorage.setItem('calendarCollapsed', 'true');
    }
}

// Función para restaurar el estado del colapso
function restoreCalendarState() {
    const container = document.getElementById('calendarContainer');
    const toggle = document.getElementById('calendarToggle');
    const icon = document.getElementById('calendarIcon');
    const text = document.getElementById('calendarToggleText');

    if (!container || !toggle || !icon || !text) return;

    const isCollapsed = localStorage.getItem('calendarCollapsed') === 'true';

    if (isCollapsed) {
        container.classList.add('collapsed');
        toggle.classList.add('collapsed');
        icon.className = 'fas fa-chevron-down';
        text.textContent = 'Mostrar';
    }
}

// Estilos CSS para toast
if (typeof toastStyles === 'undefined') {
    const toastStyles = `
<style>
.toast {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10000;
    opacity: 0;
    transform: translateX(100%);
    transition: all 0.3s ease;
}

.toast.show {
    opacity: 1;
    transform: translateX(0);
}

.toast-content {
    background: white;
    border-radius: 8px;
    padding: 16px 20px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    display: flex;
    align-items: center;
    gap: 12px;
    min-width: 300px;
    border-left: 4px solid;
}

.toast-content.success {
    border-left-color: #48bb78;
}

.toast-content.error {
    border-left-color: #e53e3e;
}

.toast-content.warning {
    border-left-color: #ed8936;
}

.toast-content.info {
    border-left-color: #4299e1;
}

.toast-content i {
    font-size: 18px;
}

.toast-content.success i {
    color: #48bb78;
}

.toast-content.error i {
    color: #e53e3e;
}

.toast-content.warning i {
    color: #ed8936;
}

.toast-content.info i {
    color: #4299e1;
}

.ripple {
    position: absolute;
    border-radius: 50%;
    background: rgba(255, 255, 255, 0.3);
    transform: scale(0);
    animation: ripple 0.6s linear;
    pointer-events: none;
}

@keyframes ripple {
    to {
        transform: scale(4);
        opacity: 0;
    }
}

.custom-tooltip {
    position: absolute;
    background: #2d3748;
    color: white;
    padding: 8px 12px;
    border-radius: 6px;
    font-size: 12px;
    z-index: 1000;
    opacity: 0;
    transition: opacity 0.2s ease;
    pointer-events: none;
    white-space: nowrap;
}

.custom-tooltip::before {
    content: '';
    position: absolute;
    top: 100%;
    left: 50%;
    transform: translateX(-50%);
    border: 5px solid transparent;
    border-top-color: #2d3748;
}
</style>
`;

    // Agregar estilos al head
    document.head.insertAdjacentHTML('beforeend', toastStyles);
}

// Funciones para el modal de estadísticas de usuario
function showUserStats(userId) {
    // Mostrar loading en el modal
    const modal = document.getElementById('userStatsModal');
    const content = document.getElementById('userStatsContent');

    modal.style.display = 'block';
    content.innerHTML = `
        <div class="empty-state">
            <i class="fas fa-spinner fa-spin"></i>
            <p>Cargando estadísticas del usuario...</p>
        </div>
    `;

    // Hacer petición AJAX para obtener estadísticas
    fetch(`?route=clan_leader/get-user-stats&user_id=${userId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayUserStats(data);
            } else {
                showErrorInModal(data.message || 'Error al cargar estadísticas');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showErrorInModal('Error de conexión al cargar estadísticas');
        });
}

function displayUserStats(data) {
    const content = document.getElementById('userStatsContent');
    const user = data.user;
    const stats = data.task_stats;

    // Actualizar título del modal
    document.getElementById('userStatsTitle').textContent = `Estadísticas de ${user.full_name}`;

    // Generar HTML del contenido
    const html = `
        <div class="user-stats-container">
            <!-- Header del usuario -->
            <div class="user-header">
                <div class="user-avatar-large" style="background-color: ${user.avatar_color}">
                    ${user.initial}
                </div>
                <div class="user-info">
                    <h3>${user.full_name}</h3>
                    <p>${user.email}</p>
                    <p><strong>Estado:</strong> ${user.is_active ? 'Activo' : 'Inactivo'}</p>
                    ${user.last_login ? `<p><strong>Último acceso:</strong> ${formatDate(user.last_login)}</p>` : ''}
                </div>
            </div>
            
            <!-- Estadísticas generales -->
            <div class="user-stats-grid">
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon completed">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <div class="stat-title">Tareas Completadas</div>
                    </div>
                    <div class="stat-value">${stats.completed_tasks}</div>
                    <div class="stat-percentage">${stats.completion_percentage}% del total</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon pending">
                            <i class="fas fa-clock"></i>
                        </div>
                        <div class="stat-title">Tareas Pendientes</div>
                    </div>
                    <div class="stat-value">${stats.pending_tasks}</div>
                    <div class="stat-percentage">${stats.total_tasks > 0 ? Math.round((stats.pending_tasks / stats.total_tasks) * 100) : 0}% del total</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon in-progress">
                            <i class="fas fa-spinner"></i>
                        </div>
                        <div class="stat-title">En Progreso</div>
                    </div>
                    <div class="stat-value">${stats.in_progress_tasks}</div>
                    <div class="stat-percentage">${stats.total_tasks > 0 ? Math.round((stats.in_progress_tasks / stats.total_tasks) * 100) : 0}% del total</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon overdue">
                            <i class="fas fa-exclamation-triangle"></i>
                        </div>
                        <div class="stat-title">Tareas Vencidas</div>
                    </div>
                    <div class="stat-value">${stats.overdue_tasks}</div>
                    <div class="stat-percentage">${stats.total_tasks > 0 ? Math.round((stats.overdue_tasks / stats.total_tasks) * 100) : 0}% del total</div>
                </div>
            </div>
            
            <!-- Tareas por proyecto -->
            ${data.tasks_by_project.length > 0 ? `
                <div class="tasks-by-project">
                    <div class="project-tasks-header">
                        <h4><i class="fas fa-project-diagram"></i> Tareas por Proyecto</h4>
                    </div>
                    <div class="project-tasks-list">
                        ${data.tasks_by_project.map(project => `
                            <div class="project-task-item">
                                <div class="task-item-header">
                                    <div class="task-item-name">${project.project_name}</div>
                                    <div class="task-item-status">${project.tasks.length} tareas</div>
                                </div>
                                <div class="task-item-meta">
                                    ${project.tasks.map(task => `
                                        <span>
                                            <i class="fas fa-tasks"></i>
                                            ${task.task_name}
                                            <span class="task-item-status ${task.status}">${getStatusText(task.status)}</span>
                                            ${task.due_date ? `<i class="fas fa-calendar"></i> ${formatDate(task.due_date)}` : ''}
                                        </span>
                                    `).join('')}
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>
            ` : `
                <div class="empty-state">
                    <i class="fas fa-tasks"></i>
                    <p>No hay tareas asignadas a este usuario</p>
                </div>
            `}
            
            <!-- Actividad reciente -->
            ${data.recent_activity.length > 0 ? `
                <div class="activity-section">
                    <div class="activity-header">
                        <h4><i class="fas fa-history"></i> Actividad Reciente</h4>
                    </div>
                    <div class="activity-list">
                        ${data.recent_activity.map(activity => `
                            <div class="activity-item">
                                <div class="activity-content">
                                    <div class="activity-text">
                                        <div class="activity-action">${getActivityText(activity)}</div>
                                        <div class="activity-task">${activity.task_name}</div>
                                    </div>
                                    <div class="activity-time">${formatDate(activity.created_at)}</div>
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>
            ` : `
                <div class="empty-state">
                    <i class="fas fa-history"></i>
                    <p>No hay actividad reciente</p>
                </div>
            `}
        </div>
    `;

    content.innerHTML = html;
}

function showErrorInModal(message) {
    const content = document.getElementById('userStatsContent');
    content.innerHTML = `
        <div class="empty-state">
            <i class="fas fa-exclamation-triangle"></i>
            <p>${message}</p>
        </div>
    `;
}

function closeUserStatsModal() {
    const modal = document.getElementById('userStatsModal');
    modal.style.display = 'none';
}

function getStatusText(status) {
    const statusMap = {
        'completed': 'Completada',
        'pending': 'Pendiente',
        'in_progress': 'En Progreso',
        'overdue': 'Vencida'
    };
    return statusMap[status] || status;
}

function getActivityText(activity) {
    const actionMap = {
        'created': 'Creó la tarea',
        'updated': 'Actualizó la tarea',
        'status_changed': 'Cambió el estado',
        'assigned': 'Fue asignado',
        'unassigned': 'Fue desasignado',
        'commented': 'Agregó un comentario',
        'completed': 'Completó la tarea',
        'reopened': 'Reabrió la tarea',
        'deleted': 'Eliminó la tarea'
    };

    let text = actionMap[activity.action_type] || activity.action_type;

    if (activity.field_name && activity.old_value !== activity.new_value) {
        text += ` (${activity.field_name}: ${activity.old_value || 'N/A'} → ${activity.new_value || 'N/A'})`;
    }

    return text;
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';

    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now - date);
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

    if (diffDays === 1) {
        return 'Hoy';
    } else if (diffDays === 2) {
        return 'Ayer';
    } else if (diffDays <= 7) {
        return `Hace ${diffDays - 1} días`;
    } else {
        return date.toLocaleDateString('es-ES', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }
}

// Cerrar modal al hacer clic fuera de él
window.addEventListener('click', function (event) {
    const modal = document.getElementById('userStatsModal');
    if (event.target === modal) {
        closeUserStatsModal();
    }
});

// Función para mostrar modales de confirmación personalizados
function showConfirmationModal(options) {
    const {
        title = 'Confirmar Acción',
        message = '¿Estás seguro de que quieres realizar esta acción?',
        icon = 'warning',
        confirmText = 'Confirmar',
        cancelText = 'Cancelar',
        onConfirm = null,
        onCancel = null,
        type = 'warning' // warning, info, success
    } = options;

    // Crear el HTML del modal
    const modalHTML = `
        <div class="confirmation-modal-overlay" id="confirmationModalOverlay">
            <div class="confirmation-modal" id="confirmationModal">
                <div class="confirmation-modal-header">
                    <h3 class="confirmation-modal-title">${title}</h3>
                </div>
                <div class="confirmation-modal-body">
                    <i class="fas fa-${getIconForType(type)} confirmation-modal-icon ${type}"></i>
                    <p class="confirmation-modal-message">${message}</p>
                    <div class="confirmation-modal-actions">
                        <button class="confirmation-modal-btn cancel" id="confirmationCancelBtn">
                            ${cancelText}
                        </button>
                        <button class="confirmation-modal-btn confirm" id="confirmationConfirmBtn">
                            ${confirmText}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Agregar el modal al DOM
    document.body.insertAdjacentHTML('beforeend', modalHTML);

    const overlay = document.getElementById('confirmationModalOverlay');
    const modal = document.getElementById('confirmationModal');
    const confirmBtn = document.getElementById('confirmationConfirmBtn');
    const cancelBtn = document.getElementById('confirmationCancelBtn');

    // Mostrar el modal con animación
    setTimeout(() => {
        overlay.classList.add('show');
    }, 10);

    // Función para cerrar el modal
    const closeModal = (result) => {
        overlay.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(overlay);
            if (result && onConfirm) {
                onConfirm();
            } else if (!result && onCancel) {
                onCancel();
            }
        }, 300);
    };

    // Event listeners
    confirmBtn.addEventListener('click', () => closeModal(true));
    cancelBtn.addEventListener('click', () => closeModal(false));

    // Cerrar al hacer clic en el overlay
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
            closeModal(false);
        }
    });

    // Cerrar con Escape
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeModal(false);
            document.removeEventListener('keydown', handleEscape);
        }
    };
    document.addEventListener('keydown', handleEscape);

    // Enfocar el botón de cancelar por defecto
    cancelBtn.focus();
}

// Función para obtener el icono según el tipo
function getIconForType(type) {
    const icons = {
        warning: 'exclamation-triangle',
        info: 'info-circle',
        success: 'check-circle',
        danger: 'times-circle',
        question: 'question-circle'
    };
    return icons[type] || 'exclamation-triangle';
}

// (Obsoleto) Versión anterior de comentarios y adjuntos eliminada para evitar duplicados.

// Función para mostrar notificaciones - DESACTIVADA para evitar duplicados
// La función showNotification ahora se maneja desde task_details.php
/*
function showNotification(message, type = 'info') {
    // Crear notificación si no existe
    let notification = document.getElementById('notification');
    if (!notification) {
        const notificationHTML = `
            <div id="notification" class="notification" style="display: none;">
                <div class="notification-content">
                    <span id="notificationMessage"></span>
                    <button onclick="closeNotification()" class="notification-close">&times;</button>
                </div>
            </div>
        `;
        document.body.insertAdjacentHTML('beforeend', notificationHTML);
        notification = document.getElementById('notification');
    }
    
    // Configurar notificación
    document.getElementById('notificationMessage').textContent = message;
    
    // Configurar estilos según tipo
    notification.className = `notification notification-${type}`;
    
    // Mostrar notificación
    notification.style.display = 'block';
    
    // Ocultar automáticamente después de 5 segundos
    setTimeout(() => {
        closeNotification();
    }, 5000);
}

// Función para cerrar notificación
function closeNotification() {
    const notification = document.getElementById('notification');
    if (notification) {
        notification.style.display = 'none';
    }
}
*/

// === Utilidades globales para adjuntos en comentarios ===
// Asegurar disponibilidad global aunque la vista no declare estas funciones
window.toggleCommentAttachments = function (button) {
    try {
        const container = button.closest('.comment-item').querySelector('.comment-attachments');
        if (container) {
            container.style.display = container.style.display === 'none' ? 'block' : 'none';
        }
    } catch (e) {
        console.error('toggleCommentAttachments error:', e);
    }
}

window.openPreview = function (url, name) {
    try {
        let modal = document.getElementById('previewModal');
        if (!modal) {
            const html = `
      <div id="previewModal" class="modal" style="display:none;">
        <div class="modal-content" style="max-width:800px;">
          <div class="modal-header">
            <h3 class="modal-title" id="previewTitle">Vista previa</h3>
            <span class="close" onclick="closePreview()">&times;</span>
          </div>
          <div id="previewBody" style="max-height:70vh;overflow:auto;"></div>
          <div class="modal-actions" style="margin-top:12px;text-align:right;">
            <a id="previewDownload" class="btn btn-primary" href="#" download>Descargar</a>
          </div>
        </div>
      </div>`;
            document.body.insertAdjacentHTML('beforeend', html);
            modal = document.getElementById('previewModal');
        }
        const body = document.getElementById('previewBody');
        const a = document.getElementById('previewDownload');
        body.innerHTML = '';
        a.href = url;
        a.setAttribute('download', name || 'archivo');
        const lower = (name || '').toLowerCase();
        if (lower.endsWith('.png') || lower.endsWith('.jpg') || lower.endsWith('.jpeg') || lower.endsWith('.gif') || lower.endsWith('.webp')) {
            const img = document.createElement('img'); img.src = url; img.style.maxWidth = '100%'; body.appendChild(img);
        } else if (lower.endsWith('.pdf')) {
            const iframe = document.createElement('iframe'); iframe.src = url; iframe.style.width = '100%'; iframe.style.height = '70vh'; body.appendChild(iframe);
        } else {
            const p = document.createElement('p'); p.textContent = 'No hay vista previa para este formato. Usa Descargar.'; body.appendChild(p);
        }
        modal.style.display = 'block';
    } catch (e) {
        console.error('openPreview error:', e);
    }
}

window.closePreview = function () {
    const modal = document.getElementById('previewModal');
    if (modal) modal.style.display = 'none';
}

// Estilos CSS para modales de confirmación
if (!document.getElementById('confirmation-modal-styles')) {
    const confirmationModalStyles = document.createElement('style');
    confirmationModalStyles.id = 'confirmation-modal-styles';
    confirmationModalStyles.textContent = `
    .confirmation-modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 10000;
        opacity: 0;
        transition: opacity 0.3s ease;
    }

    .confirmation-modal-overlay.show {
        opacity: 1;
    }

    .confirmation-modal {
        background: white;
        border-radius: 12px;
        padding: 0;
        max-width: 400px;
        width: 90%;
        box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        transform: scale(0.9);
        transition: transform 0.3s ease;
    }

    .confirmation-modal-overlay.show .confirmation-modal {
        transform: scale(1);
    }

    .confirmation-modal-header {
        padding: 20px 24px 0;
    }

    .confirmation-modal-title {
        margin: 0;
        font-size: 1.25rem;
        font-weight: 600;
        color: #1f2937;
    }

    .confirmation-modal-body {
        padding: 20px 24px 24px;
        text-align: center;
    }

    .confirmation-modal-icon {
        font-size: 3rem;
        margin-bottom: 1rem;
    }

    .confirmation-modal-icon.warning {
        color: #f59e0b;
    }

    .confirmation-modal-icon.info {
        color: #3b82f6;
    }

    .confirmation-modal-icon.success {
        color: #10b981;
    }

    .confirmation-modal-icon.danger {
        color: #ef4444;
    }

    .confirmation-modal-message {
        margin: 0 0 1.5rem 0;
        color: #6b7280;
        line-height: 1.5;
        white-space: pre-line;
    }

    .confirmation-modal-actions {
        display: flex;
        gap: 12px;
        justify-content: center;
    }

    .confirmation-modal-btn {
        padding: 10px 20px;
        border: none;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        min-width: 100px;
    }

    .confirmation-modal-btn.cancel {
        background: #f3f4f6;
        color: #374151;
    }

    .confirmation-modal-btn.cancel:hover {
        background: #e5e7eb;
    }

    .confirmation-modal-btn.confirm {
        background: #ef4444;
        color: white;
    }

    .confirmation-modal-btn.confirm:hover {
        background: #dc2626;
    }

    .confirmation-modal-btn.confirm.warning {
        background: #f59e0b;
    }

    .confirmation-modal-btn.confirm.warning:hover {
        background: #d97706;
    }

    .confirmation-modal-btn.confirm.success {
        background: #10b981;
    }

    .confirmation-modal-btn.confirm.success:hover {
        background: #059669;
    }

    .confirmation-modal-btn.confirm.info {
        background: #3b82f6;
    }

    .confirmation-modal-btn.confirm.info:hover {
        background: #2563eb;
    }

    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    `;

    // Agregar estilos al head
    document.head.appendChild(confirmationModalStyles);
}

// Variables globales para task details
let selectedFiles = [];
let selectedUsers = [];
let isCommentSubmitting = false;

function setCommentSubmitting(isLoading) {
    const form = document.querySelector('.add-comment-form');
    const sendBtn = document.querySelector('.add-comment-form .btn.btn-primary');
    const textarea = document.getElementById('newComment');
    const fileInput = document.getElementById('fileAttachment');
    const attachBtn = document.querySelector('.add-comment-form .btn-attachment');
    if (!sendBtn) return;

    if (isLoading) {
        isCommentSubmitting = true;
        // Guardar HTML original para restaurar después
        if (!sendBtn.dataset.originalHtml) {
            sendBtn.dataset.originalHtml = sendBtn.innerHTML;
        }
        sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Enviando...';
        sendBtn.disabled = true;
        if (textarea) textarea.disabled = true;
        if (fileInput) fileInput.disabled = true;
        if (attachBtn) attachBtn.classList.add('disabled');
        if (form) form.style.opacity = '0.7';
    } else {
        isCommentSubmitting = false;
        if (sendBtn.dataset.originalHtml) {
            sendBtn.innerHTML = sendBtn.dataset.originalHtml;
        }
        sendBtn.disabled = false;
        if (textarea) textarea.disabled = false;
        if (fileInput) fileInput.disabled = false;
        if (attachBtn) attachBtn.classList.remove('disabled');
        if (form) form.style.opacity = '';
    }
}

// Función para eliminar tarea
function deleteTask(taskId) {
    showConfirmationModal({
        title: 'Confirmar Eliminación',
        message: '¿Estás seguro de que quieres eliminar esta tarea?',
        type: 'warning',
        confirmText: 'Eliminar',
        cancelText: 'Cancelar',
        onConfirm: () => {
            fetch('?route=clan_leader/delete-task', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'task_id=' + taskId
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Tarea eliminada exitosamente', 'success');
                        setTimeout(() => {
                            window.location.href = '?route=clan_leader/tasks';
                        }, 1000);
                    } else {
                        showNotification('Error al eliminar la tarea: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Error al eliminar la tarea', 'error');
                });
        }
    });
}

// Función para agregar comentario con progreso de subida
function addComment() {
    if (isCommentSubmitting) return;
    const commentTextarea = document.getElementById('newComment');
    const commentText = (commentTextarea ? commentTextarea.value : '').trim();
    if (!commentText) {
        showNotification('Por favor escribe un comentario', 'error');
        return;
    }

    const container = document.querySelector('.task-details-container');
    const taskId = container ? container.dataset.taskId : null;
    if (!taskId) {
        showNotification('No se pudo obtener el ID de la tarea', 'error');
        return;
    }

    const formData = new FormData();
    formData.append('task_id', taskId);
    formData.append('comment_text', commentText);
    if (selectedFiles && selectedFiles.length > 0) {
        selectedFiles.forEach(file => formData.append('attachments[]', file));
    }

    // Crear/asegurar UI de progreso
    let progressWrap = document.getElementById('uploadProgressWrap');
    if (!progressWrap) {
        const form = document.querySelector('.add-comment-form');
        if (form) {
            form.insertAdjacentHTML('beforeend', `
        < div id = "uploadProgressWrap" style = "margin-top:8px; display:none;" >
                  <div style="height:6px;background:#e5e7eb;border-radius:999px;overflow:hidden;">
                    <div id="uploadProgressBar" style="height:6px;width:0%;background:#3b82f6;transition:width .2s ease;"></div>
                  </div>
                  <div id="uploadProgressText" style="margin-top:6px;font-size:12px;color:#6b7280;">Preparando subida...</div>
                </div >
        `);
            progressWrap = document.getElementById('uploadProgressWrap');
        }
    }
    const progressBar = document.getElementById('uploadProgressBar');
    const progressText = document.getElementById('uploadProgressText');
    if (progressWrap && progressBar && progressText) {
        progressBar.style.width = '0%';
        progressText.textContent = 'Iniciando subida...';
        progressWrap.style.display = 'block';
    }

    setCommentSubmitting(true);

    // Usar XHR para obtener progreso de subida
    const xhr = new XMLHttpRequest();
    xhr.open('POST', '?route=clan_leader/add-task-comment', true);

    xhr.upload.onprogress = function (e) {
        if (e.lengthComputable && progressBar && progressText) {
            const percent = Math.round((e.loaded / e.total) * 100);
            progressBar.style.width = percent + '%';
            progressText.textContent = `Subiendo archivos... ${percent}% `;
        }
    };

    xhr.onreadystatechange = function () {
        if (xhr.readyState === 4) {
            try {
                const data = JSON.parse(xhr.responseText || '{}');
                if (xhr.status >= 200 && xhr.status < 300 && data.success) {
                    if (Array.isArray(data.attachments_saved)) {
                        const savedCount = data.attachments_saved.length;
                        const receivedCount = Array.isArray(data.attachments_received) ? data.attachments_received.length : savedCount;
                        const msg = savedCount > 0 ? `Comentario y ${savedCount}/${receivedCount} adjuntos guardados` : 'Comentario agregado';
                        showNotification(msg, 'success');
                    } else {
                        showNotification('Comentario agregado exitosamente', 'success');
                    }
                    if (commentTextarea) commentTextarea.value = '';
                    removeAttachment();
                    setTimeout(() => location.reload(), 800);
                } else {
                    const message = (data && data.message) ? data.message : 'Error al agregar comentario';
                    showNotification(message, 'error');
                }
            } catch (err) {
                showNotification('Error de respuesta del servidor', 'error');
            } finally {
                setCommentSubmitting(false);
                if (progressWrap) progressWrap.style.display = 'none';
            }
        }
    };

    xhr.onerror = function () {
        showNotification('Error de red al enviar el comentario', 'error');
        setCommentSubmitting(false);
        if (progressWrap) progressWrap.style.display = 'none';
    };

    xhr.send(formData);
}

// Función para manejar adjuntos (múltiples)
function handleFileAttachment(input) {
    if (input.files && input.files.length > 0) {
        // Acumular archivos seleccionados en múltiples clics
        const incoming = Array.from(input.files);
        const existingKeys = new Set(selectedFiles.map(f => `${f.name}|${f.size}|${f.lastModified || 0}`));
        incoming.forEach(f => {
            const key = `${f.name}|${f.size}|${f.lastModified || 0}`;
            if (!existingKeys.has(key)) {
                selectedFiles.push(f);
                existingKeys.add(key);
            }
        });

        // Limpiar el input para permitir volver a seleccionar el mismo archivo si se desea
        try { input.value = ''; } catch (e) { /* ignore */ }

        const nameSpan = document.getElementById('attachmentName');
        const preview = document.getElementById('attachmentPreview');
        if (nameSpan) {
            if (selectedFiles.length === 1) {
                nameSpan.textContent = selectedFiles[0].name;
            } else {
                const previewNames = selectedFiles.slice(0, 3).map(f => f.name).join(', ');
                const more = selectedFiles.length > 3 ? ` y ${selectedFiles.length - 3} más` : '';
                nameSpan.textContent = `${selectedFiles.length} archivos: ${previewNames}${more}`;
            }
        }
        if (preview) preview.style.display = 'block';
    }
}

// Función para remover adjuntos
function removeAttachment() {
    selectedFiles = [];
    const preview = document.getElementById('attachmentPreview');
    const input = document.getElementById('fileAttachment');
    if (preview) preview.style.display = 'none';
    if (input) input.value = '';
}

// Función para abrir modal de agregar tarea desde el calendario
function openAddTaskModal(selectedDate) {
    // Cerrar el modal actual de tareas
    closeTaskModal();

    // Mostrar el modal de creación de tareas
    showCreateTaskModal(selectedDate);
}

// Función para mostrar el modal de creación de tareas
function showCreateTaskModal(selectedDate) {
    // Crear el modal si no existe
    let modal = document.getElementById('createTaskModal');
    if (!modal) {
        createTaskModalHTML();
        modal = document.getElementById('createTaskModal');
    }

    // Establecer la fecha preseleccionada
    const dateInput = document.getElementById('createTaskDueDate');
    const recurrenceStartInput = document.getElementById('createTaskRecurrenceStartDate');

    if (selectedDate) {
        // Establecer en fecha límite (para tareas normales)
        if (dateInput) {
            dateInput.value = selectedDate;
        }

        // También establecer en fecha de inicio de recurrencia (para tareas recurrentes)
        if (recurrenceStartInput) {
            recurrenceStartInput.value = selectedDate;
        }
    }

    // Mostrar el modal
    modal.style.display = 'flex';
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
}

// Función para crear el HTML del modal de creación de tareas
function createTaskModalHTML() {
    const modalHTML = `
        <div id="createTaskModal" class="create-task-modal" style="display: none;">
            <div class="create-task-modal-content">
                <div class="create-task-modal-header">
                    <h3>
                        <i class="fas fa-plus-circle"></i>
                        Crear Nueva Tarea
                    </h3>
                    <button class="create-task-modal-close" onclick="closeCreateTaskModal()">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <div class="create-task-modal-body">
                    <form id="createTaskForm">
                        <!-- Título de la tarea -->
                        <div class="form-group">
                            <label for="createTaskTitle">Título de la tarea *</label>
                            <input type="text" id="createTaskTitle" name="task_title" placeholder="Título de la tarea *" required>
                        </div>
                        
                        <!-- Fecha límite y Proyecto -->
                        <div class="form-row">
                            <div class="form-group">
                                <label for="createTaskDueDate">Fecha límite *</label>
                                <div class="date-input-wrapper">
                                    <input type="date" id="createTaskDueDate" name="task_due_date" required>
                                    <i class="fas fa-calendar-alt"></i>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label for="createTaskProject">Proyecto/Concepto</label>
                                <div class="select-wrapper">
                                    <select id="createTaskProject" name="task_project">
                                        <option value="">Seleccionar proyecto...</option>
                                    </select>
                                    <i class="fas fa-chevron-down"></i>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Configuración de recurrencia -->
                        <div class="form-group">
                            <div class="checkbox-container">
                                <input type="checkbox" id="createTaskIsRecurrent" name="is_recurrent" value="1" onchange="toggleCreateTaskRecurrenceFields()">
                                <label for="createTaskIsRecurrent" class="checkbox-label">
                                    <i class="fas fa-redo"></i>
                                    Tarea Recurrente
                                </label>
                            </div>
                            <small class="field-help">Las tareas recurrentes se crearán automáticamente en el rango de fechas especificado</small>
                        </div>
                        
                        <div id="createTaskRecurrenceFields" class="recurrence-fields" style="display: none;">
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="createTaskRecurrenceType">Tipo de Recurrencia *</label>
                                    <div class="select-wrapper">
                                        <select id="createTaskRecurrenceType" name="recurrence_type">
                                            <option value="">Seleccionar...</option>
                                            <option value="daily">Diaria</option>
                                            <option value="weekly">Semanal</option>
                                            <option value="monthly">Mensual</option>
                                        </select>
                                        <i class="fas fa-chevron-down"></i>
                                    </div>
                                </div>
                                <div class="form-group">
                                    <label for="createTaskRecurrenceStartDate">Fecha de Inicio *</label>
                                    <div class="date-input-wrapper">
                                        <input type="date" id="createTaskRecurrenceStartDate" name="recurrence_start_date">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                </div>
                            </div>
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="createTaskRecurrenceEndDate">Fecha de Vigencia (Opcional)</label>
                                    <div class="date-input-wrapper">
                                        <input type="date" id="createTaskRecurrenceEndDate" name="recurrence_end_date">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                    <small class="field-help">Si no se especifica, la recurrencia será indefinida. Las tareas se crearán desde la fecha de inicio hasta esta fecha.</small>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Descripción -->
                        <div class="form-group">
                            <label for="createTaskDescription">Descripción</label>
                            <textarea id="createTaskDescription" name="task_description" rows="3" placeholder="Descripción de la tarea..."></textarea>
                        </div>
                        
                        <!-- Prioridad -->
                        <div class="form-group">
                            <label for="createTaskPriority">Prioridad</label>
                            <div class="select-wrapper">
                                <select id="createTaskPriority" name="priority">
                                    <option value="">Seleccionar prioridad...</option>
                                    <option value="low">Baja</option>
                                    <option value="medium">Media</option>
                                    <option value="high">Alta</option>
                                    <option value="critical">Crítica</option>
                                </select>
                                <i class="fas fa-chevron-down"></i>
                            </div>
                        </div>
                        
                        <!-- Asignar a colaboradores -->
                        <div class="form-group">
                            <label for="createTaskAssignedMembers">Asignar a colaboradores *</label>
                            <div class="select-wrapper">
                                <select id="createTaskAssignedMembers" name="assigned_members[]" multiple required>
                                    <!-- Los colaboradores se cargarán dinámicamente -->
                                </select>
                                <i class="fas fa-chevron-down"></i>
                            </div>
                            <small class="field-help">Mantén presionado Ctrl (Cmd en Mac) para seleccionar múltiples colaboradores</small>
                        </div>
                    </form>
                </div>
                
                <div class="create-task-modal-footer">
                    <button type="button" class="btn-cancel" onclick="closeCreateTaskModal()">
                        <i class="fas fa-times"></i>
                        Cancelar
                    </button>
                    <button type="button" class="btn-create" onclick="submitCreateTask()">
                        <i class="fas fa-plus"></i>
                        Crear Tarea
                    </button>
                </div>
            </div>
        </div>
    `;

    document.body.insertAdjacentHTML('beforeend', modalHTML);

    // Cargar datos necesarios
    loadCreateTaskData();
}

// Función para cargar datos necesarios para el modal
function loadCreateTaskData() {
    console.log('🔄 Cargando datos para el modal de creación de tareas...');

    // Cargar proyectos
    fetch('?route=clan_leader/get-projects-for-modal')
        .then(response => {
            console.log('📡 Respuesta de proyectos:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('📊 Datos de proyectos recibidos:', data);
            if (data.success && data.projects) {
                const projectSelect = document.getElementById('createTaskProject');
                if (projectSelect) {
                    console.log('✅ Cargando', data.projects.length, 'proyectos en el select');
                    data.projects.forEach(project => {
                        const option = document.createElement('option');
                        option.value = project.project_id;
                        option.textContent = project.project_name;
                        projectSelect.appendChild(option);
                    });
                } else {
                    console.error('❌ No se encontró el select de proyectos');
                }
            } else {
                console.error('❌ Error en datos de proyectos:', data.message);
            }
        })
        .catch(error => {
            console.error('💥 Error cargando proyectos:', error);
        });

    // Cargar colaboradores
    fetch('?route=clan_leader/get-collaborators-for-modal')
        .then(response => {
            console.log('📡 Respuesta de colaboradores:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('👥 Datos de colaboradores recibidos:', data);
            if (data.success && data.collaborators) {
                const collaboratorsSelect = document.getElementById('createTaskAssignedMembers');
                if (collaboratorsSelect) {
                    console.log('✅ Cargando', data.collaborators.length, 'colaboradores en el select');
                    collaboratorsSelect.innerHTML = '';
                    data.collaborators.forEach(collaborator => {
                        const option = document.createElement('option');
                        option.value = collaborator.user_id;
                        option.textContent = collaborator.full_name;
                        collaboratorsSelect.appendChild(option);
                    });
                } else {
                    console.error('❌ No se encontró el select de colaboradores');
                }
            } else {
                console.error('❌ Error en datos de colaboradores:', data.message);
            }
        })
        .catch(error => {
            console.error('💥 Error cargando colaboradores:', error);
        });
}

// Función para obtener color del colaborador
function getCollaboratorColor(userId) {
    const colors = ['#667eea', '#48bb78', '#ed8936', '#e53e3e', '#9f7aea', '#38b2ac', '#a0aec0', '#f6e05e'];
    return colors[userId % colors.length];
}

// Función para mostrar/ocultar campos de recurrencia en el modal de creación
function toggleCreateTaskRecurrenceFields() {
    const checkbox = document.getElementById('createTaskIsRecurrent');
    const fields = document.getElementById('createTaskRecurrenceFields');
    const dueDateField = document.getElementById('createTaskDueDate');
    const recurrenceStartField = document.getElementById('createTaskRecurrenceStartDate');
    const dueDateGroup = dueDateField ? dueDateField.closest('.form-group') : null;

    if (checkbox.checked) {
        // Mostrar campos de recurrencia
        fields.style.display = 'block';

        // Ocultar campo de fecha límite cuando es recurrente
        if (dueDateGroup) {
            dueDateGroup.style.display = 'none';
        }

        // Quitar required y limpiar valor del campo de fecha límite
        if (dueDateField) {
            dueDateField.required = false;
            dueDateField.removeAttribute('required');
            // NO limpiar el valor, mantenerlo para referencia
        }

        // Si la fecha de inicio de recurrencia está vacía, copiar la fecha límite
        if (recurrenceStartField && !recurrenceStartField.value && dueDateField && dueDateField.value) {
            recurrenceStartField.value = dueDateField.value;
        }

        // Hacer requeridos los campos de recurrencia
        document.getElementById('createTaskRecurrenceType').required = true;
        document.getElementById('createTaskRecurrenceStartDate').required = true;
    } else {
        // Ocultar campos de recurrencia
        fields.style.display = 'none';

        // Mostrar campo de fecha límite normal
        if (dueDateGroup) {
            dueDateGroup.style.display = 'block';
        }

        // Restaurar required
        if (dueDateField) {
            dueDateField.required = true;
            dueDateField.setAttribute('required', 'required');
        }

        // Quitar requerimiento de campos de recurrencia
        document.getElementById('createTaskRecurrenceType').required = false;
        document.getElementById('createTaskRecurrenceStartDate').required = false;

        // Limpiar valores de recurrencia
        document.getElementById('createTaskRecurrenceType').value = '';
        document.getElementById('createTaskRecurrenceStartDate').value = '';
        document.getElementById('createTaskRecurrenceEndDate').value = '';
    }
}


// Función para cerrar el modal de creación de tareas
function closeCreateTaskModal() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.classList.remove('show');
        setTimeout(() => {
            modal.style.display = 'none';
            // Limpiar formulario
            document.getElementById('createTaskForm').reset();
        }, 300);
    }
}

// Función para enviar el formulario de creación de tareas
function submitCreateTask() {
    const form = document.getElementById('createTaskForm');
    const formData = new FormData();

    // Validar campos requeridos
    const title = document.getElementById('createTaskTitle').value.trim();
    const dueDate = document.getElementById('createTaskDueDate').value;
    const isRecurrent = document.getElementById('createTaskIsRecurrent').checked;
    const recurrenceStart = document.getElementById('createTaskRecurrenceStartDate').value;
    const assignedMembersSelect = document.getElementById('createTaskAssignedMembers');
    const selectedMembers = Array.from(assignedMembersSelect.selectedOptions);

    if (!title) {
        showToast('Por favor ingresa el título de la tarea', 'error');
        return;
    }

    // Validación de fecha según tipo de tarea
    if (isRecurrent) {
        if (!recurrenceStart) {
            showToast('Por favor ingresa la fecha de inicio de recurrencia', 'error');
            return;
        }
    } else {
        if (!dueDate) {
            showToast('Por favor selecciona una fecha límite', 'error');
            return;
        }
    }

    if (selectedMembers.length === 0) {
        showToast('Debes asignar al menos un colaborador', 'error');
        return;
    }

    // Recopilar datos del formulario
    formData.append('task_title', title);
    formData.append('task_due_date', dueDate || '');
    formData.append('task_project', document.getElementById('createTaskProject').value);
    formData.append('task_description', document.getElementById('createTaskDescription').value);
    formData.append('priority', document.getElementById('createTaskPriority').value);

    // Agregar campos de recurrencia si aplica
    if (isRecurrent) {
        formData.append('is_recurrent', '1');
        formData.append('recurrence_type', document.getElementById('createTaskRecurrenceType').value);
        formData.append('recurrence_start_date', recurrenceStart);
        formData.append('recurrence_end_date', document.getElementById('createTaskRecurrenceEndDate').value || '');
    }

    // Agregar miembros asignados
    selectedMembers.forEach(member => {
        formData.append('assigned_members[]', member.value);
    });

    // Agregar subtareas vacías
    formData.append('subtasks', JSON.stringify([]));

    // Deshabilitar botón mientras se envía
    const createBtn = document.querySelector('.btn-create');
    const originalText = createBtn.innerHTML;
    createBtn.disabled = true;
    createBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando...';

    // Enviar datos al servidor
    fetch('?route=clan_leader/create-task', {
        method: 'POST',
        body: formData
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Tarea creada exitosamente', 'success');
                closeCreateTaskModal();
                // Recargar la página para mostrar la nueva tarea
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            } else {
                showToast(data.message || 'Error al crear la tarea', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('Error al crear la tarea', 'error');
        })
        .finally(() => {
            // Restaurar botón
            createBtn.disabled = false;
            createBtn.innerHTML = originalText;
        });
}

// Event listeners para el modal de creación de tareas
document.addEventListener('DOMContentLoaded', function () {
    // Cerrar modal al hacer clic fuera de él
    document.addEventListener('click', function (e) {
        const modal = document.getElementById('createTaskModal');
        if (modal && e.target === modal) {
            closeCreateTaskModal();
        }
    });

    // Cerrar modal con tecla Escape
    document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape') {
            const modal = document.getElementById('createTaskModal');
            if (modal && modal.style.display !== 'none') {
                closeCreateTaskModal();
            }
        }
    });
});

// Fin del archivo