#!/bin/bash

# Script de despliegue automático para RinoTrack
# Servidor: bitnami@52.3.160.124
# Directorio remoto: /home/bitnami/htdocs/rinotrackalfa

set -e  # Salir si hay algún error

# Configuración
REMOTE_USER="bitnami"
REMOTE_HOST="52.3.160.124"
REMOTE_PATH="/home/bitnami/htdocs/rinotrackalfa"
LOCAL_PATH="/Users/abdiel/Documents/Proyectos/htdocs/rinotrackalfa"
PEM_FILE="$LOCAL_PATH/rinotrack.pem"

# Colores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}🚀 Iniciando despliegue de RinoTrack...${NC}"

# Verificar que el archivo PEM existe
if [ ! -f "$PEM_FILE" ]; then
    echo -e "${RED}❌ Error: No se encuentra el archivo rinotrack.pem${NC}"
    exit 1
fi

# Verificar permisos del archivo PEM
chmod 400 "$PEM_FILE"

# Función para ejecutar comandos SSH
ssh_exec() {
    ssh -i "$PEM_FILE" -o StrictHostKeyChecking=no "$REMOTE_USER@$REMOTE_HOST" "$1"
}

# Función para copiar archivos con rsync
rsync_deploy() {
    rsync -avz --delete \
        -e "ssh -i $PEM_FILE -o StrictHostKeyChecking=no" \
        --exclude='.git' \
        --exclude='node_modules' \
        --exclude='.DS_Store' \
        --exclude='*.log' \
        --exclude='rinotrack.pem' \
        --exclude='deploy.sh' \
        --exclude='.env' \
        "$LOCAL_PATH/" "$REMOTE_USER@$REMOTE_HOST:$REMOTE_PATH/"
}

echo -e "${YELLOW}📋 Verificando conexión SSH...${NC}"
if ssh_exec "echo 'Conexión SSH exitosa'"; then
    echo -e "${GREEN}✅ Conexión SSH establecida${NC}"
else
    echo -e "${RED}❌ Error: No se pudo establecer conexión SSH${NC}"
    exit 1
fi

echo -e "${YELLOW}📁 Creando directorio remoto si no existe...${NC}"
ssh_exec "mkdir -p $REMOTE_PATH"

echo -e "${YELLOW}📤 Sincronizando archivos...${NC}"
rsync_deploy

echo -e "${YELLOW}🔧 Configurando permisos en servidor remoto...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    
    # Permisos para directorios
    find . -type d -exec chmod 755 {} \;
    
    # Permisos para archivos PHP
    find . -name '*.php' -exec chmod 644 {} \;
    
    # Permisos especiales para uploads - usar grupo daemon para PHP-FPM
    mkdir -p public/uploads/task_attachments
    sudo chown -R bitnami:daemon public/uploads
    sudo chmod -R 775 public/uploads
    
    # Permisos para archivos de configuración
    chmod 644 config/*.php
    
    # Hacer ejecutable el directorio public
    chmod 755 public
"

echo -e "${YELLOW}⚙️ Actualizando configuración para producción...${NC}"

# Crear archivo de configuración para producción
cat > /tmp/app_production.php << 'EOF'
<?php
/**
 * Configuración de la aplicación - PRODUCCIÓN
 */

// Configuración de la aplicación
define('APP_NAME', 'Polaris');
define('APP_URL', 'http://52.3.160.124/rinotrackalfa/public/');
define('APP_DEBUG', false); // Desactivar debug en producción

// Configuración de zona horaria
date_default_timezone_set('America/Mexico_City');

// Configuración de sesión
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 0); // HTTP por ahora
ini_set('session.cookie_samesite', 'Lax');

// Configuración de errores (producción)
if (APP_DEBUG) {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    error_reporting(0);
    // Log de errores
    ini_set('log_errors', 1);
    ini_set('error_log', '/tmp/rinotrack_errors.log');
}

// Incluir archivos de configuración
require_once __DIR__ . '/database.php';
require_once __DIR__ . '/mail.php';

// Iniciar sesión si no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
EOF

# Copiar configuración de producción
scp -i "$PEM_FILE" -o StrictHostKeyChecking=no /tmp/app_production.php "$REMOTE_USER@$REMOTE_HOST:$REMOTE_PATH/config/app.php"

echo -e "${YELLOW}🗄️ Configurando base de datos para producción...${NC}"

# Crear configuración de base de datos para producción
cat > /tmp/database_production.php << 'EOF'
<?php
/**
 * Configuración de base de datos - PRODUCCIÓN
 */

define('DB_HOST', 'localhost');
define('DB_USERNAME', 'root');
define('DB_PASSWORD', ''); // Configurar según el servidor
define('DB_NAME', 'rinotrack');

/**
 * Clase para manejar la conexión a la base de datos
 */
class Database {
    private static $connection = null;
    
    /**
     * Obtener conexión PDO singleton
     */
    public static function getConnection() {
        if (self::$connection === null) {
            try {
                $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
                $options = [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                ];
                
                self::$connection = new PDO($dsn, DB_USERNAME, DB_PASSWORD, $options);
            } catch (PDOException $e) {
                // En producción, registrar el error en lugar de mostrarlo
                error_log('Error de conexión a la base de datos: ' . $e->getMessage());
                die('Error de conexión a la base de datos. Contacte al administrador.');
            }
        }
        
        return self::$connection;
    }
}
EOF

# Copiar configuración de base de datos
scp -i "$PEM_FILE" -o StrictHostKeyChecking=no /tmp/database_production.php "$REMOTE_USER@$REMOTE_HOST:$REMOTE_PATH/config/database.php"

echo -e "${YELLOW}🌐 Configurando servidor web...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    
    # Crear .htaccess en la raíz si no existe
    if [ ! -f .htaccess ]; then
        cat > .htaccess << 'HTACCESS_EOF'
RewriteEngine On
RewriteRule ^(.*)$ public/\$1 [L]
HTACCESS_EOF
    fi
    
    # Verificar .htaccess en public
    if [ ! -f public/.htaccess ]; then
        cat > public/.htaccess << 'HTACCESS_PUBLIC_EOF'
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php?route=\$1 [QSA,L]
HTACCESS_PUBLIC_EOF
    fi
"

echo -e "${YELLOW}🔍 Verificando instalación...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    echo 'Estructura de directorios:'
    ls -la
    echo ''
    echo 'Contenido de public:'
    ls -la public/
    echo ''
    echo 'Archivos de configuración:'
    ls -la config/
"

# Limpiar archivos temporales
rm -f /tmp/app_production.php /tmp/database_production.php

echo -e "${GREEN}✅ Despliegue completado exitosamente!${NC}"
echo -e "${BLUE}🌐 La aplicación debería estar disponible en: http://52.3.160.124/rinotrackalfa/public/${NC}"
echo -e "${YELLOW}📝 Notas importantes:${NC}"
echo -e "   • Configurar la contraseña de la base de datos en el servidor"
echo -e "   • Importar la base de datos si es necesario"
echo -e "   • Verificar que Apache/Nginx esté configurado correctamente"
echo -e "   • Revisar logs en caso de errores: /tmp/rinotrack_errors.log"
