#!/bin/bash

# Script de despliegue a DESARROLLO para RinoTrack
# Servidor: bitnami@52.3.160.124
# Directorio remoto: /home/bitnami/htdocs/desarrollo/rinotrack

set -e  # Salir si hay algún error

# Configuración
REMOTE_USER="bitnami"
REMOTE_HOST="52.3.160.124"
REMOTE_PATH="/home/bitnami/htdocs/desarrollo/rinotrack"
LOCAL_PATH="/Users/abdiel/Documents/Proyectos/htdocs/rinotrackalfa"
PEM_FILE="$LOCAL_PATH/rinotrack.pem"

# Colores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}🚀 Iniciando despliegue COMPLETO de RinoTrack a DESARROLLO...${NC}"

# Verificar que el archivo PEM existe
if [ ! -f "$PEM_FILE" ]; then
    echo -e "${RED}❌ Error: No se encuentra el archivo rinotrack.pem${NC}"
    exit 1
fi

# Verificar permisos del archivo PEM
chmod 400 "$PEM_FILE"

# Función para ejecutar comandos SSH
ssh_exec() {
    ssh -i "$PEM_FILE" -o StrictHostKeyChecking=no "$REMOTE_USER@$REMOTE_HOST" "$1"
}

# Función para copiar archivos con rsync
rsync_deploy() {
    rsync -avz --delete \
        -e "ssh -i $PEM_FILE -o StrictHostKeyChecking=no" \
        --exclude='.git' \
        --exclude='node_modules' \
        --exclude='.DS_Store' \
        --exclude='*.log' \
        --exclude='rinotrack.pem' \
        --exclude='deploy.sh' \
        --exclude='auto-deploy.sh' \
        --exclude='deploy-desarrollo.sh' \
        --exclude='.env' \
        "$LOCAL_PATH/" "$REMOTE_USER@$REMOTE_HOST:$REMOTE_PATH/"
}

echo -e "${YELLOW}📋 Verificando conexión SSH...${NC}"
if ssh_exec "echo 'Conexión SSH exitosa'"; then
    echo -e "${GREEN}✅ Conexión SSH establecida${NC}"
else
    echo -e "${RED}❌ Error: No se pudo establecer conexión SSH${NC}"
    exit 1
fi

echo -e "${YELLOW}📁 Creando directorios remotos...${NC}"
ssh_exec "mkdir -p /home/bitnami/htdocs/desarrollo"
ssh_exec "mkdir -p $REMOTE_PATH"

echo -e "${YELLOW}📤 Sincronizando archivos (esto puede tomar unos minutos)...${NC}"
rsync_deploy

echo -e "${YELLOW}🔧 Configurando permisos en servidor remoto...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    
    # Permisos para directorios
    find . -type d -exec chmod 755 {} \;
    
    # Permisos para archivos PHP
    find . -name '*.php' -exec chmod 644 {} \;
    
    # Permisos especiales para uploads - usar grupo daemon para PHP-FPM
    mkdir -p public/uploads/task_attachments
    
    # Cambiar propietario y grupo (bitnami:daemon)
    sudo chown -R bitnami:daemon public/uploads
    
    # Directorios: 775 (rwxrwxr-x) para que daemon pueda crear archivos
    sudo chmod -R 775 public/uploads
    
    # Archivos: 664 (rw-rw-r--) para que daemon pueda escribir
    sudo find public/uploads -type f -exec chmod 664 {} \;
    
    # Permisos para archivos de configuración
    chmod 644 config/*.php 2>/dev/null || true
    
    # Hacer ejecutable el directorio public
    chmod 755 public
"

echo -e "${YELLOW}⚙️ Configurando rutas para DESARROLLO...${NC}"

# Crear archivo de configuración para desarrollo
cat > /tmp/app_desarrollo.php << 'EOF'
<?php
/**
 * Configuración de la aplicación - DESARROLLO
 */

// Configuración de la aplicación
define('APP_NAME', 'Polaris');
define('APP_URL', 'https://rinotrack.rinorisk.com/desarrollo/rinotrack/public/');
define('APP_DEBUG', false); // Desactivar debug en producción

// Configuración de zona horaria
date_default_timezone_set('America/Mexico_City');

// Configuración de sesión
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 1); // HTTPS habilitado
ini_set('session.cookie_samesite', 'Lax');

// Configuración de errores
if (APP_DEBUG) {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    error_reporting(0);
    // Log de errores
    ini_set('log_errors', 1);
    ini_set('error_log', '/tmp/rinotrack_desarrollo_errors.log');
}

// Incluir archivos de configuración
require_once __DIR__ . '/database.php';
require_once __DIR__ . '/mail.php';

// Iniciar sesión si no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
EOF

# Copiar configuración de desarrollo
scp -i "$PEM_FILE" -o StrictHostKeyChecking=no /tmp/app_desarrollo.php "$REMOTE_USER@$REMOTE_HOST:$REMOTE_PATH/config/app.php"

echo -e "${YELLOW}🌐 Configurando .htaccess para DESARROLLO...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    
    # Crear .htaccess en la raíz
    cat > .htaccess << 'HTACCESS_EOF'
# RinoTrack - Redirección a public/
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Redirigir todo el tráfico a public/
    RewriteRule ^$ public/ [L]
    RewriteRule (.*) public/\$1 [L]
</IfModule>

# Protección de archivos sensibles
<FilesMatch \"\\.(env|pem|sh|md)$\">
    Order allow,deny
    Deny from all
</FilesMatch>
HTACCESS_EOF
    
    # Crear .htaccess en public
    cat > public/.htaccess << 'HTACCESS_PUBLIC_EOF'
# RinoTrack Public - Enrutamiento
<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Establecer la base de reescritura para esta ruta
    RewriteBase /desarrollo/rinotrack/public/
    
    # Si el archivo o directorio existe, mostrarlo directamente
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    
    # Si no existe, redirigir a index.php
    RewriteRule ^(.*)$ index.php?route=\$1 [QSA,L]
</IfModule>

# Desactivar listado de directorios
Options -Indexes

# Protección adicional
<FilesMatch \"\\.(env|pem|sh|md)$\">
    Order allow,deny
    Deny from all
</FilesMatch>

# Permitir acceso a assets
<FilesMatch \"\\.(css|js|png|jpg|jpeg|gif|svg|woff|woff2|ttf|eot|ico)$\">
    Order allow,deny
    Allow from all
</FilesMatch>
HTACCESS_PUBLIC_EOF
    
    echo 'Archivos .htaccess creados correctamente'
"

echo -e "${YELLOW}🔍 Verificando instalación...${NC}"
ssh_exec "
    cd $REMOTE_PATH
    echo '=== Estructura de directorios ==='
    ls -la | head -20
    echo ''
    echo '=== Contenido de public ==='
    ls -la public/ | head -15
    echo ''
    echo '=== Archivos de configuración ==='
    ls -la config/ 2>/dev/null || echo 'Directorio config no encontrado'
    echo ''
    echo '=== Verificando permisos de uploads ==='
    ls -la public/uploads/ 2>/dev/null || echo 'Directorio uploads no encontrado'
"

# Limpiar archivos temporales
rm -f /tmp/app_desarrollo.php

echo -e "${GREEN}✅ Despliegue COMPLETO a DESARROLLO exitoso!${NC}"
echo -e "${BLUE}🌐 La aplicación está disponible en:${NC}"
echo -e "   • https://rinotrack.rinorisk.com/desarrollo/rinotrack/public/ (Producción)"
echo -e "   • http://52.3.160.124/desarrollo/rinotrack/public/ (IP directa)"
echo -e "${YELLOW}📝 Configuración aplicada:${NC}"
echo -e "   • Ruta: /home/bitnami/htdocs/desarrollo/rinotrack"
echo -e "   • URL base: https://rinotrack.rinorisk.com/desarrollo/rinotrack/public/"
echo -e "   • RewriteBase configurado: /desarrollo/rinotrack/public/"
echo -e "   • Permisos de uploads configurados con grupo daemon"
echo -e "   • HTTPS habilitado para cookies"
echo -e ""
echo -e "${YELLOW}⚠️ Verificar:${NC}"
echo -e "   • Que la base de datos esté configurada correctamente"
echo -e "   • Que Apache tenga mod_rewrite habilitado"
echo -e "   • Que AllowOverride esté en 'All' para este directorio"

