<?php

class User {
    private $db;
    
    public function __construct() {
        $this->db = Database::getConnection();
    }
    
    /**
     * Buscar usuario por username o email
     */
    public function findByUsernameOrEmail($username) {
        try {
            // Intentar con la columna phone primero
            $stmt = $this->db->prepare("
                SELECT user_id, username, email, phone, password_hash, full_name, is_active, last_login, created_at, avatar_path 
                FROM Users 
                WHERE (username = ? OR email = ?) AND is_active = 1
            ");
            $stmt->execute([$username, $username]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            // Si falla (columna phone no existe), intentar sin phone
            try {
                error_log("Intentando consulta sin columna phone: " . $e->getMessage());
                $stmt = $this->db->prepare("
                    SELECT user_id, username, email, password_hash, full_name, is_active, last_login, created_at, avatar_path 
                    FROM Users 
                    WHERE (username = ? OR email = ?) AND is_active = 1
                ");
                $stmt->execute([$username, $username]);
                return $stmt->fetch();
            } catch (PDOException $e2) {
                error_log("Error al buscar usuario: " . $e2->getMessage());
                return false;
            }
        }
    }
    
    /**
     * Buscar usuario por ID
     */
    public function findById($id) {
        try {
            // Intentar con la columna phone primero
            $stmt = $this->db->prepare("
                SELECT user_id, username, email, phone, full_name, is_active, last_login, created_at, avatar_path 
                FROM Users 
                WHERE user_id = ? AND is_active = 1
            ");
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            // Si falla (columna phone no existe), intentar sin phone
            try {
                $stmt = $this->db->prepare("
                    SELECT user_id, username, email, full_name, is_active, last_login, created_at, avatar_path 
                    FROM Users 
                    WHERE user_id = ? AND is_active = 1
                ");
                $stmt->execute([$id]);
                return $stmt->fetch();
            } catch (PDOException $e2) {
                error_log("Error al buscar usuario por ID: " . $e2->getMessage());
                return false;
            }
        }
    }
    
    /**
     * Obtener todos los usuarios del sistema
     */
    public function getAll() {
        try {
            $stmt = $this->db->prepare("
                SELECT user_id, username, email, phone, full_name, is_active, last_login, created_at, avatar_path 
                FROM Users 
                ORDER BY full_name ASC, username ASC
            ");
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error al obtener todos los usuarios: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Buscar usuario por ID sin importar su estado activo
     */
    public function findByIdAnyStatus($id) {
        try {
            $stmt = $this->db->prepare("
                SELECT user_id, username, email, phone, full_name, is_active, last_login, created_at, avatar_path 
                FROM Users 
                WHERE user_id = ?
            ");
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error al buscar usuario por ID (cualquier estado): " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Actualizar último login del usuario
     */
    public function updateLastLogin($userId) {
        try {
            $stmt = $this->db->prepare("UPDATE Users SET last_login = NOW() WHERE user_id = ?");
            return $stmt->execute([$userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar último login: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Crear nuevo usuario
     */
    public function create($username, $email, $password, $fullName, $phone = null) {
        try {
            error_log("User::create - Starting user creation");
            error_log("Parameters: username='$username', email='$email', fullName='$fullName', phone='$phone', password_length=" . strlen($password));
            
            $passwordHash = password_hash($password, PASSWORD_DEFAULT);
            error_log("Password hash generated successfully: " . (empty($passwordHash) ? "FAILED" : "SUCCESS"));
            
            $sql = "INSERT INTO Users (username, email, phone, password_hash, full_name, avatar_path, created_at) VALUES (?, ?, ?, ?, ?, '', NOW())";
            error_log("SQL query: $sql");
            
            $stmt = $this->db->prepare($sql);
            if (!$stmt) {
                error_log("PREPARE FAILED: " . print_r($this->db->errorInfo(), true));
                return false;
            }
            
            $result = $stmt->execute([$username, $email, $phone, $passwordHash, $fullName]);
            error_log("Execute result: " . ($result ? "SUCCESS" : "FAILED"));
            
            if (!$result) {
                error_log("EXECUTE ERROR: " . print_r($stmt->errorInfo(), true));
                return false;
            }
            
            $lastId = $this->db->lastInsertId();
            error_log("Last insert ID: $lastId");
            
            return $lastId;
        } catch (PDOException $e) {
            error_log("PDOException in User::create: " . $e->getMessage());
            error_log("Error Code: " . $e->getCode());
            error_log("SQL State: " . ($e->errorInfo[0] ?? 'N/A'));
            return false;
        } catch (Exception $e) {
            error_log("General Exception in User::create: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verificar si un username o email ya existe
     */
    public function exists($username, $email) {
        try {
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count 
                FROM Users 
                WHERE username = ? OR email = ?
            ");
            $stmt->execute([$username, $email]);
            $result = $stmt->fetch();
            return $result['count'] > 0;
        } catch (PDOException $e) {
            error_log("Error al verificar existencia de usuario: " . $e->getMessage());
            return true; // En caso de error, asumir que existe para evitar duplicados
        }
    }
    
    /**
     * Verificar si un username o email ya existe excluyendo un usuario específico
     */
    public function existsExcludingUser($excludeUserId, $username, $email) {
        try {
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count 
                FROM Users 
                WHERE (username = ? OR email = ?) AND user_id != ?
            ");
            $stmt->execute([$username, $email, $excludeUserId]);
            $result = $stmt->fetch();
            return $result['count'] > 0;
        } catch (PDOException $e) {
            error_log("Error al verificar existencia de usuario (excluyendo): " . $e->getMessage());
            return true; // En caso de error, asumir que existe para evitar duplicados
        }
    }
    
    /**
     * Activar/desactivar usuario
     */
    public function setActive($userId, $active = true) {
        try {
            $stmt = $this->db->prepare("UPDATE Users SET is_active = ? WHERE user_id = ?");
            return $stmt->execute([$active ? 1 : 0, $userId]);
        } catch (PDOException $e) {
            error_log("Error al cambiar estado de usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtener todos los usuarios con información de rol
     */
    public function getAllWithRoles() {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    u.*,
                    r.role_name,
                    r.role_id
                FROM Users u
                LEFT JOIN User_Roles ur ON u.user_id = ur.user_id
                LEFT JOIN Roles r ON ur.role_id = r.role_id
                ORDER BY u.created_at DESC
            ");
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error al obtener usuarios con roles: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Actualizar información del usuario
     */
    public function update($userId, $username, $email, $fullName, $isActive = 1, $phone = null) {
        try {
            $stmt = $this->db->prepare("
                UPDATE Users 
                SET username = ?, email = ?, phone = ?, full_name = ?, is_active = ?
                WHERE user_id = ?
            ");
            return $stmt->execute([$username, $email, $phone, $fullName, $isActive, $userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Actualizar solo el estado del usuario
     */
    public function updateStatus($userId, $isActive) {
        try {
            $stmt = $this->db->prepare("
                UPDATE Users 
                SET is_active = ?
                WHERE user_id = ?
            ");
            return $stmt->execute([$isActive, $userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar estado del usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Cambiar contraseña del usuario
     */
    public function updatePassword($userId, $newPassword) {
        try {
            $passwordHash = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $this->db->prepare("UPDATE Users SET password_hash = ? WHERE user_id = ?");
            return $stmt->execute([$passwordHash, $userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar contraseña: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Actualizar contraseña en texto plano (según requerimiento específico)
     */
    public function updatePasswordPlain($userId, $newPassword) {
        try {
            $stmt = $this->db->prepare("UPDATE Users SET password_hash = ? WHERE user_id = ?");
            return $stmt->execute([$newPassword, $userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar contraseña (plain): " . $e->getMessage());
            return false;
        }
    }

    /**
     * Actualizar avatar_path del usuario
     */
    public function updateAvatarPath($userId, $avatarPath) {
        try {
            $stmt = $this->db->prepare("UPDATE Users SET avatar_path = ? WHERE user_id = ?");
            return $stmt->execute([$avatarPath, $userId]);
        } catch (PDOException $e) {
            error_log("Error al actualizar avatar_path: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Eliminar usuario
     */
    public function delete($userId) {
        try {
            $stmt = $this->db->prepare("DELETE FROM Users WHERE user_id = ?");
            return $stmt->execute([$userId]);
        } catch (PDOException $e) {
            error_log("Error al eliminar usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtener estadísticas de usuarios
     */
    public function getStats() {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    COUNT(*) as total_users,
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_users,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_users,
                    SUM(CASE WHEN last_login >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as recent_users
                FROM Users
            ");
            $stmt->execute();
            $result = $stmt->fetch();
            
            // Asegurar que todos los valores sean numéricos
            return [
                'total_users' => (int)($result['total_users'] ?? 0),
                'active_users' => (int)($result['active_users'] ?? 0),
                'inactive_users' => (int)($result['inactive_users'] ?? 0),
                'recent_users' => (int)($result['recent_users'] ?? 0)
            ];
        } catch (PDOException $e) {
            error_log("Error al obtener estadísticas de usuarios: " . $e->getMessage());
            return [
                'total_users' => 0,
                'active_users' => 0,
                'inactive_users' => 0,
                'recent_users' => 0
            ];
        }
    }
    
    /**
     * Buscar usuarios por término
     */
    public function search($searchTerm) {
        try {
            $searchTerm = '%' . $searchTerm . '%';
            $stmt = $this->db->prepare("
                SELECT 
                    u.*,
                    r.role_name
                FROM Users u
                LEFT JOIN User_Roles ur ON u.user_id = ur.user_id
                LEFT JOIN Roles r ON ur.role_id = r.role_id
                WHERE u.username LIKE ? 
                   OR u.email LIKE ? 
                   OR u.full_name LIKE ?
                ORDER BY u.full_name
            ");
            $stmt->execute([$searchTerm, $searchTerm, $searchTerm]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error al buscar usuarios: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtener el clan del usuario (mantiene compatibilidad, retorna el clan activo)
     */
    public function getUserClan($userId) {
        // Primero intentar obtener el clan activo
        $activeClan = $this->getActiveClan($userId);
        if ($activeClan) {
            return $activeClan;
        }
        
        // Si no tiene clan activo, obtener el PRIMER clan al que se unió (su departamento original)
        return $this->getUserMainClan($userId);
    }

    /**
     * Obtener el clan PRINCIPAL del usuario (el primero al que se unió)
     * Independientemente de cuál tenga activo en este momento.
     */
    public function getUserMainClan($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    c.clan_id,
                    c.clan_name,
                    c.clan_departamento
                FROM Clans c
                JOIN Clan_Members cm ON c.clan_id = cm.clan_id
                WHERE cm.user_id = ?
                ORDER BY cm.clan_member_id ASC
                LIMIT 1
            ");
            $stmt->execute([$userId]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error al obtener clan principal del usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtener todos los clanes del usuario
     */
    public function getUserClans($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    c.clan_id,
                    c.clan_name,
                    c.clan_departamento,
                    c.created_at
                FROM Clans c
                JOIN Clan_Members cm ON c.clan_id = cm.clan_id
                WHERE cm.user_id = ?
                ORDER BY c.clan_name
            ");
            $stmt->execute([$userId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error al obtener clanes del usuario: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtener el clan activo del usuario
     */
    public function getActiveClan($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    c.clan_id,
                    c.clan_name,
                    c.clan_departamento
                FROM Clans c
                JOIN User_Active_Clan uac ON c.clan_id = uac.active_clan_id
                WHERE uac.user_id = ?
            ");
            $stmt->execute([$userId]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Error al obtener clan activo del usuario: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Establecer el clan activo del usuario
     */
    public function setActiveClan($userId, $clanId) {
        try {
            error_log("setActiveClan: userId=$userId, clanId=$clanId");
            
            // Verificar que el usuario pertenece al clan
            $stmt = $this->db->prepare("
                SELECT COUNT(*) as count 
                FROM Clan_Members 
                WHERE user_id = ? AND clan_id = ?
            ");
            $stmt->execute([$userId, $clanId]);
            $result = $stmt->fetch();
            
            error_log("setActiveClan: Usuario pertenece al clan? count=" . $result['count']);
            
            if ($result['count'] == 0) {
                error_log("Error: Usuario $userId no pertenece al clan $clanId");
                
                // Listar clanes del usuario para debug
                $debugStmt = $this->db->prepare("SELECT clan_id FROM Clan_Members WHERE user_id = ?");
                $debugStmt->execute([$userId]);
                $userClans = $debugStmt->fetchAll(PDO::FETCH_COLUMN);
                error_log("Clanes del usuario $userId: " . implode(', ', $userClans));
                
                return false;
            }
            
            // Primero, intentar crear la tabla si no existe
            try {
                $this->db->exec("
                    CREATE TABLE IF NOT EXISTS User_Active_Clan (
                        user_id INT NOT NULL,
                        active_clan_id INT NOT NULL,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        PRIMARY KEY (user_id),
                        FOREIGN KEY (user_id) REFERENCES Users(user_id) ON DELETE CASCADE,
                        FOREIGN KEY (active_clan_id) REFERENCES Clans(clan_id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                ");
                error_log("setActiveClan: Tabla User_Active_Clan verificada/creada");
            } catch (PDOException $e) {
                error_log("setActiveClan: Advertencia al crear tabla: " . $e->getMessage());
            }
            
            // Actualizar o insertar clan activo
            $stmt = $this->db->prepare("
                INSERT INTO User_Active_Clan (user_id, active_clan_id) 
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE active_clan_id = ?, updated_at = NOW()
            ");
            $result = $stmt->execute([$userId, $clanId, $clanId]);
            
            error_log("setActiveClan: Resultado de actualización: " . ($result ? 'SUCCESS' : 'FAILED'));
            
            return $result;
        } catch (PDOException $e) {
            error_log("Error al establecer clan activo: " . $e->getMessage());
            error_log("Stack trace: " . $e->getTraceAsString());
            return false;
        }
    }
    
    /**
     * Obtener usuarios por clan
     */
    public function getUsersByClan($clanId) {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    u.user_id,
                    u.username,
                    u.full_name,
                    u.email,
                    u.phone,
                    u.is_active,
                    u.last_login,
                    u.created_at,
                    r.role_name
                FROM Users u
                JOIN Clan_Members cm ON u.user_id = cm.user_id
                LEFT JOIN User_Roles ur ON u.user_id = ur.user_id
                LEFT JOIN Roles r ON ur.role_id = r.role_id
                WHERE cm.clan_id = ?
                ORDER BY u.full_name
            ");
            $stmt->execute([$clanId]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Error al obtener usuarios por clan: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtener todos los usuarios del sistema
     */
    public function getAllUsers() {
        try {
            // Primero obtener usuarios básicos
            $stmt = $this->db->prepare("
                SELECT 
                    u.user_id,
                    u.username,
                    u.full_name,
                    u.email,
                    u.phone,
                    u.is_active,
                    u.last_login,
                    u.created_at
                FROM Users u
                WHERE u.is_active = 1
                ORDER BY u.full_name
            ");
            $stmt->execute();
            $users = $stmt->fetchAll();
            
            // Luego obtener roles y clanes para cada usuario
            foreach ($users as &$user) {
                // Obtener rol
                $roleStmt = $this->db->prepare("
                    SELECT r.role_name 
                    FROM User_Roles ur 
                    JOIN Roles r ON ur.role_id = r.role_id 
                    WHERE ur.user_id = ?
                ");
                $roleStmt->execute([$user['user_id']]);
                $role = $roleStmt->fetch();
                $user['role_name'] = $role ? $role['role_name'] : null;
                
                // Obtener clan
                $clanStmt = $this->db->prepare("
                    SELECT c.clan_name 
                    FROM Clan_Members cm 
                    JOIN Clans c ON cm.clan_id = c.clan_id 
                    WHERE cm.user_id = ?
                ");
                $clanStmt->execute([$user['user_id']]);
                $clan = $clanStmt->fetch();
                $user['clan_name'] = $clan ? $clan['clan_name'] : null;
            }
            
            return $users;
        } catch (PDOException $e) {
            error_log("Error al obtener todos los usuarios: " . $e->getMessage());
            return [];
        }
    }
}