<?php
// Iniciar buffer de salida
ob_start();

// Verificar que tenemos los datos necesarios
if (!isset($project) || !isset($tasks)) {
    echo '<div style="padding: 20px; text-align: center;">Error: Datos del proyecto no disponibles</div>';
    $content = ob_get_clean();
    require_once __DIR__ . '/../layout.php';
    return;
}

$projectName = htmlspecialchars($project['project_name'] ?? 'Proyecto');
?>

<!-- Cargar CSS Monday Theme -->
<link rel="stylesheet" href="<?= APP_URL ?>/assets/css/monday-theme.css">
<script src="https://unpkg.com/@lottiefiles/dotlottie-wc@0.8.5/dist/dotlottie-wc.js" type="module"></script>

<style>
/* ===== ESTILOS EXACTOS DE CLAN LEADER PROJECT TASKS ===== */

/* Sección principal */
.all-tasks-section {
    background: var(--monday-surface, #ffffff);
    border-radius: 8px;
    padding: 24px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
    border: 1px solid var(--monday-border-color, #d0d4e4);
}

.section-header {
    margin-bottom: 20px;
}

.section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--monday-text-color, #323338);
    margin: 0;
}

/* Tarjetas de estadísticas */
.stats-cards-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}

.stat-card {
    background: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    border: 1px solid var(--monday-border-color, #d0d4e4);
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12);
}

.stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    color: white;
    background: #3b82f6;
    flex-shrink: 0;
}

.stat-icon.total { background: #3b82f6; }
.stat-icon.completed { background: #10b981; }
.stat-icon.in-progress { background: #3b82f6; }
.stat-icon.pending { background: #f59e0b; }

.stat-content { flex: 1; }

.stat-value {
    font-size: 32px;
    font-weight: 700;
    color: #1f2937;
    line-height: 1;
    margin-bottom: 4px;
}

.stat-label {
    font-size: 12px;
    font-weight: 600;
    color: #6b7280;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Header con botón */
.section-header-with-action {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 16px;
}

.header-content-wrapper {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    width: 100%;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 16px;
}

.header-content-left {
    flex: 1;
    min-width: 200px;
}

.header-content-right {
    display: flex;
    align-items: center;
    gap: 16px;
}

.project-title-header {
    font-size: 24px;
    font-weight: 700;
    color: var(--monday-text-color, #323338);
    margin: 0 0 8px 0;
    display: flex;
    align-items: center;
    gap: 8px;
}

.project-icon-header {
    color: #0073ea;
    font-size: 20px;
}

.project-description-box {
    margin-top: 12px;
    padding: 12px 16px;
    background: #f8f9fa;
    border-radius: 8px;
    border-left: 4px solid #0073ea;
}

.project-description-box p {
    font-size: 14px;
    color: #676879;
    line-height: 1.6;
    margin: 0;
}

.header-actions-wrapper {
    display: flex;
    align-items: center;
    gap: 16px;
}

.lottie-animation-container {
    display: flex;
    justify-content: center;
    align-items: center;
}

.btn-header-action {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    font-size: 14px;
    cursor: pointer;
    transition: all 0.2s;
}

.btn-proyectos {
    background: #f1f5f9;
    color: #475569;
    border: 1px solid #e2e8f0;
}

.btn-proyectos:hover {
    background: #e2e8f0;
    border-color: #cbd5e1;
    transform: translateY(-1px);
}

.btn-nueva-tarea {
    background: linear-gradient(135deg, #059669 0%, #10b981 100%);
    color: white !important;
    border: none;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
}

.btn-nueva-tarea:hover {
    background: linear-gradient(135deg, #047857 0%, #059669 100%);
    transform: translateY(-2px);
    color: white !important;
    box-shadow: 0 8px 20px rgba(16, 185, 129, 0.4);
}

/* Filtros estilo Monday */
.filters-container {
    margin-bottom: 20px;
}

.filter-row {
    display: flex;
    flex-direction: row;
    flex-wrap: nowrap;
    gap: 16px;
    align-items: center;
    margin-bottom: 12px;
}

.filter-item {
    display: inline-flex;
    flex-direction: row;
    align-items: center;
    gap: 8px;
    white-space: nowrap;
}

.filter-item label {
    font-size: 13px;
    font-weight: 500;
    color: var(--monday-text-secondary, #676879);
}

.filter-item select {
    padding: 8px 12px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    color: var(--monday-text-color, #323338);
    background: white;
    min-width: 140px;
    cursor: pointer;
}

.filter-item select:hover,
.filter-item select:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
}

.btn-reset-filters {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 14px;
    background: transparent;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    font-weight: 500;
    color: var(--monday-text-secondary, #676879);
    cursor: pointer;
}

.btn-reset-filters:hover {
    background: #f5f6f8;
    border-color: #c4c4c4;
}

/* Búsqueda */
.search-row {
    display: block;
    max-width: 400px;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: var(--monday-text-secondary, #676879);
    font-size: 14px;
}

.search-input {
    width: 100%;
    padding: 8px 12px 8px 36px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    color: var(--monday-text-color, #323338);
}

.search-input:hover,
.search-input:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
    box-shadow: 0 0 0 2px rgba(0, 115, 234, 0.1);
}

/* Tabla estilo Monday */
.tasks-table-container {
    overflow-x: auto;
    margin-top: 16px;
}

.tasks-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.tasks-table thead {
    background: #f5f6f8;
    border-bottom: 1px solid var(--monday-border-color, #d0d4e4);
}

.tasks-table th {
    padding: 12px 16px;
    text-align: left;
    font-weight: 600;
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: var(--monday-text-secondary, #676879);
    white-space: nowrap;
}

.tasks-table tbody tr {
    border-bottom: 1px solid #eaebef;
    transition: background-color 0.15s;
}

.tasks-table tbody tr:hover {
    background: rgba(0, 115, 234, 0.04);
}

.tasks-table tbody tr.completed {
    background: rgba(0, 200, 117, 0.05);
}

.tasks-table td {
    padding: 14px 16px;
    color: var(--monday-text-color, #323338);
    vertical-align: middle;
}

/* Columnas específicas */
.th-priority, .td-priority { width: 90px; white-space: nowrap; }
.th-task, .td-task { width: 30%; min-width: 200px; }
.th-assigned, .td-assigned { width: 15%; min-width: 120px; }
.th-due-date, .td-due-date { width: 110px; }
.th-status, .td-status { width: 120px; white-space: nowrap; }
.th-actions, .td-actions { width: 130px; text-align: center; }

/* Badge de prioridad */
.priority-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    white-space: nowrap;
}

.priority-badge.priority-critical,
.priority-badge.priority-urgent {
    background: #ffeef0;
    color: #e2445c;
}

.priority-badge.priority-high {
    background: #fff3e0;
    color: #fdab3d;
}

.priority-badge.priority-medium {
    background: #e5f4ff;
    color: #0073ea;
}

.priority-badge.priority-low {
    background: #e8f5e9;
    color: #00c875;
}

/* Info de tarea */
.task-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.task-title {
    font-weight: 500;
    color: var(--monday-text-color, #323338);
}

.task-description {
    font-size: 12px;
    color: var(--monday-text-secondary, #676879);
    max-width: 350px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.assigned-user {
    color: #0073ea;
    font-weight: 500;
    font-size: 13px;
}

.no-assigned {
    color: var(--monday-text-secondary, #676879);
    font-size: 12px;
    font-style: italic;
}

/* Fecha de vencimiento */
.due-date-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.due-date-text {
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 4px;
    color: var(--monday-text-color, #323338);
    font-size: 13px;
}

.due-date-text i {
    font-size: 11px;
    color: var(--monday-text-secondary, #676879);
}

.no-due-date {
    color: var(--monday-text-secondary, #676879);
    font-size: 12px;
    font-style: italic;
}

/* Badge de estado */
.status-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 4px 10px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    white-space: nowrap;
}

.status-badge.status-pending {
    background: #c4c4c4;
    color: white;
}

.status-badge.status-in_progress {
    background: #fdab3d;
    color: white;
}

.status-badge.status-completed {
    background: #00c875;
    color: white;
}

/* Select de estado */
.status-select {
    padding: 6px 10px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 12px;
    background: white;
    color: var(--monday-text-color, #323338);
    font-weight: 600;
    cursor: pointer;
    min-width: 120px;
}

.status-select:hover,
.status-select:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
}

/* Botones de acción */
.actions-group {
    display: flex;
    gap: 4px;
    justify-content: center;
    align-items: center;
}

.btn-action-table {
    display: inline-flex !important;
    align-items: center !important;
    justify-content: center !important;
    width: 32px !important;
    height: 32px !important;
    border: none !important;
    border-radius: 6px !important;
    font-size: 13px !important;
    cursor: pointer !important;
    text-decoration: none !important;
    background: #669AE1 !important;
    color: white !important;
    transition: all 0.2s !important;
}

.btn-action-table:hover {
    background: #4a7bc8 !important;
    transform: translateY(-1px) !important;
}

.btn-action-table i,
.btn-action-table .fas,
.btn-action-table .fa {
    color: white !important;
}

/* Paginación */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px 0;
    margin-top: 16px;
    border-top: 1px solid var(--monday-border-color, #d0d4e4);
}

.pagination-info {
    font-size: 13px;
    color: var(--monday-text-secondary, #676879);
}

.pagination-buttons {
    display: flex;
    gap: 4px;
    align-items: center;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 32px;
    height: 32px;
    padding: 0 10px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    background: white;
    color: var(--monday-text-color, #323338);
    font-size: 13px;
    cursor: pointer;
    transition: all 0.2s;
}

.pagination-btn:hover:not(:disabled):not(.active) {
    background: #0073ea;
    border-color: #0073ea;
    color: white;
}

.pagination-btn.active {
    background: #0073ea;
    border-color: #0073ea;
    color: white;
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Estado vacío */
.empty-state {
    text-align: center;
    padding: 60px 20px;
}

.empty-state .empty-icon {
    font-size: 48px;
    color: var(--monday-text-secondary, #676879);
    margin-bottom: 16px;
}

.empty-state h3 {
    font-size: 18px;
    color: var(--monday-text-color, #323338);
    margin: 0 0 8px 0;
}

.empty-state p {
    color: var(--monday-text-secondary, #676879);
    margin: 0 0 20px 0;
}

/* ========== ESTILOS MODAL MONDAY.COM ========== */
@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes scaleIn {
    from { 
        opacity: 0;
        transform: scale(0.95) translateY(10px);
    }
    to { 
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(15, 23, 42, 0.6);
    backdrop-filter: blur(10px);
    -webkit-backdrop-filter: blur(10px);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.modal-content {
    background: rgba(252, 252, 253, 0.9);
    backdrop-filter: blur(40px) saturate(105%);
    -webkit-backdrop-filter: blur(40px) saturate(105%);
    border-radius: 20px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow: hidden;
    box-shadow: 
        0 8px 32px rgba(16, 185, 129, 0.04),
        0 0 0 1px rgba(255, 255, 255, 0.4) inset,
        0 1px 2px rgba(0, 0, 0, 0.05);
    border: 1px solid rgba(236, 253, 245, 0.08);
    animation: scaleIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    display: flex;
    flex-direction: column;
}

.modal-header {
    padding: 24px 28px;
    border-bottom: 1px solid rgba(236, 253, 245, 0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(to bottom, rgba(252, 252, 253, 0.6), rgba(236, 253, 245, 0.08));
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    position: relative;
    z-index: 1;
    flex-shrink: 0;
}

.modal-header > div {
    display: flex;
    align-items: center;
    gap: 12px;
    flex: 1;
    min-width: 0;
}

.modal-header h3 {
    flex: 1;
    min-width: 0;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
    background: linear-gradient(135deg, #1e293b 0%, #475569 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    letter-spacing: -0.02em;
}

.modal-close {
    background: rgba(239, 68, 68, 0.1);
    border: none;
    border-radius: 10px;
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    color: #dc2626;
    font-size: 18px;
    flex-shrink: 0;
}

.modal-close:hover {
    background: rgba(239, 68, 68, 0.2);
    transform: scale(1.05);
}

.modal-body {
    padding: 28px;
    overflow-y: auto;
    position: relative;
    z-index: 1;
    background: rgba(252, 252, 253, 0.4);
    flex: 1;
    min-height: 0;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 500;
    color: #334155;
    font-size: 0.9rem;
}

.form-control {
    width: 100%;
    padding: 12px 16px;
    border: 2px solid rgba(203, 213, 225, 0.5);
    border-radius: 10px;
    font-size: 0.95rem;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    background: rgba(255, 255, 255, 0.8);
    color: #1e293b;
}

.form-control:focus {
    outline: none;
    border-color: #10b981;
    box-shadow: 0 0 0 3px rgba(16, 185, 129, 0.1);
    background: white;
}

.checkbox-label-modern {
    display: flex;
    align-items: center;
    gap: 10px;
    cursor: pointer;
    user-select: none;
}

.checkbox-label-modern input[type="checkbox"] {
    width: 20px;
    height: 20px;
    cursor: pointer;
}

.field-help {
    display: block;
    margin-top: 6px;
    font-size: 0.8rem;
    color: #64748b;
}

.modal-footer {
    padding: 20px 28px 24px;
    border-top: 1px solid rgba(236, 253, 245, 0.1);
    display: flex;
    gap: 12px;
    justify-content: flex-end;
    background: linear-gradient(to top, rgba(252, 252, 253, 0.6), rgba(236, 253, 245, 0.08));
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    position: relative;
    z-index: 1;
    flex-shrink: 0;
}

.btn-secondary {
    padding: 12px 24px;
    background: rgba(148, 163, 184, 0.1);
    border: 2px solid rgba(148, 163, 184, 0.2);
    border-radius: 10px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    display: inline-flex;
    align-items: center;
    gap: 8px;
    color: #475569;
    font-size: 0.9rem;
}

.btn-secondary:hover {
    background: rgba(148, 163, 184, 0.15);
    transform: translateY(-1px);
}

.btn-primary {
    padding: 12px 24px;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border: none;
    border-radius: 10px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    display: inline-flex;
    align-items: center;
    gap: 8px;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.2);
    font-size: 0.9rem;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 16px rgba(16, 185, 129, 0.3);
}

.form-errors {
    background: rgba(239, 68, 68, 0.1);
    color: #dc2626;
    padding: 12px;
    border-left: 4px solid #ef4444;
    border-radius: 8px;
    margin-top: 16px;
    font-size: 0.9rem;
}

.recurrence-fields {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-top: 1rem;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
}
</style>

<div class="all-tasks-section">
    <div class="header-content-wrapper">
        <div class="header-content-left">
            <h1 class="project-title-header">
                <i class="fas fa-folder project-icon-header"></i>
                <?= $projectName ?>
            </h1>
            <?php if (!empty($project['description'])): ?>
                <?php
                $fullDescription = htmlspecialchars($project['description']);
                $shortDescription = htmlspecialchars(mb_substr($project['description'], 0, 100));
                $displayDescription = (mb_strlen($project['description']) > 100) ? $shortDescription . '...' : $fullDescription;
                ?>
                <div class="project-description-box" title="<?= $fullDescription ?>">
                    <p><?= $displayDescription ?></p>
                </div>
            <?php endif; ?>
        </div>
        <div class="header-content-right">
            <div class="header-actions-wrapper">
                <a href="?route=clan_member/projects" class="btn-header-action btn-proyectos">
                    <i class="fas fa-folder"></i>
                    Proyectos
                </a>
                <button type="button" class="btn-header-action btn-nueva-tarea" onclick="openCreateTaskModal()">
                    <i class="fas fa-plus"></i>
                    Nueva Tarea
                </button>
            </div>
        </div>
    </div>
    
    <!-- Tarjetas de estadísticas -->
    <div class="stats-cards-container">
        <div class="stat-card">
            <div class="stat-icon total">
                <i class="fas fa-list"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-total"><?= count($tasks) ?></div>
                <div class="stat-label">Total Tareas</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon completed">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-completed"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'completed'; })) ?></div>
                <div class="stat-label">Completadas</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon in-progress">
                <i class="fas fa-spinner"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-in-progress"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'in_progress'; })) ?></div>
                <div class="stat-label">En Progreso</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon pending">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-pending"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'pending'; })) ?></div>
                <div class="stat-label">Pendientes</div>
            </div>
        </div>
    </div>
    
    <!-- Animación Lottie -->
    <div class="lottie-animation-container" style="margin: 20px 0;">
        <dotlottie-wc src="https://lottie.host/46d0b721-b84d-49de-be3a-59f0c95d2a4f/4EVkAXm90o.lottie" style="width: 160px;height: 160px" autoplay loop></dotlottie-wc>
    </div>
    
    <?php if (isset($tasks) && !empty($tasks)): ?>
        <!-- Filtros y búsqueda -->
        <div class="filters-container">
            <div class="filter-row">
                <div class="filter-item">
                    <label for="status_filter">Estado:</label>
                    <select id="status_filter" onchange="filterTasks()">
                        <option value="">Todos</option>
                        <option value="pending">Pendiente</option>
                        <option value="in_progress">En Progreso</option>
                        <option value="completed">Completado</option>
                    </select>
                </div>
                
                <div class="filter-item">
                    <label for="priority_filter">Prioridad:</label>
                    <select id="priority_filter" onchange="filterTasks()">
                        <option value="">Todas</option>
                        <option value="low">Baja</option>
                        <option value="medium">Media</option>
                        <option value="high">Alta</option>
                        <option value="urgent">Urgente</option>
                        <option value="critical">Crítica</option>
                    </select>
                </div>
                
                <div class="filter-item">
                    <label for="perPage">Mostrar:</label>
                    <select id="perPage" onchange="changePerPage()">
                        <option value="5" selected>5 por página</option>
                        <option value="10">10 por página</option>
                        <option value="25">25 por página</option>
                        <option value="50">50 por página</option>
                        <option value="all">Todos</option>
                    </select>
                </div>
                
                <button type="button" class="btn-reset-filters" onclick="resetFilters()">
                    <i class="fas fa-undo"></i>
                    Limpiar
                </button>
            </div>
            
            <div class="search-row">
                <div class="search-input-wrapper">
                    <i class="fas fa-search search-icon"></i>
                    <input type="text" 
                           id="searchInput"
                           placeholder="Buscar tareas..."
                           class="search-input"
                           oninput="debounceSearch()">
                </div>
            </div>
        </div>
        
        <!-- Tabla de Tareas -->
        <div class="tasks-table-container">
            <table class="tasks-table">
                <thead>
                    <tr>
                        <th class="th-priority">Prioridad</th>
                        <th class="th-task">Tarea</th>
                        <th class="th-assigned">Asignado</th>
                        <th class="th-due-date">Fecha Límite</th>
                        <th class="th-status">Estado</th>
                        <th class="th-actions">Acciones</th>
                    </tr>
                </thead>
                <tbody id="tasksTableBody">
                    <?php foreach ($tasks as $task): ?>
                        <?php 
                        $userId = (int)($user['user_id'] ?? 0);
                        $assignedIds = array_filter(explode(',', (string)($task['all_assigned_user_ids'] ?? '')));
                        $canEditTask = in_array((string)$userId, $assignedIds, true) || (int)($task['assigned_to_user_id'] ?? 0) === $userId || (int)($task['created_by_user_id'] ?? 0) === $userId;
                        
                        $status = $task['status'] ?? 'pending';
                        $priority = $task['priority'] ?? 'medium';
                        
                        // Prioridad texto
                        $priorityText = $priority === 'critical' ? 'Urgente' : ($priority === 'urgent' ? 'Urgente' : ($priority === 'high' ? 'Alta' : ($priority === 'low' ? 'Baja' : 'Media')));
                        ?>
                        <tr class="task-row priority-<?= $priority ?> <?= ($status === 'completed') ? 'completed' : '' ?>" 
                            data-status="<?= $status ?>" 
                            data-priority="<?= $priority ?>">
                            <td class="td-priority">
                                <span class="priority-badge priority-<?= $priority ?>">
                                    <?= $priorityText ?>
                                </span>
                            </td>
                            <td class="td-task">
                                <div class="task-info">
                                    <div class="task-title"><?= htmlspecialchars($task['task_name']) ?></div>
                                    <?php if (!empty($task['description'])): ?>
                                    <div class="task-description"><?= htmlspecialchars(substr($task['description'], 0, 80)) ?><?= strlen($task['description']) > 80 ? '...' : '' ?></div>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td class="td-assigned">
                                <?php if (!empty($task['assigned_to_fullname'])): ?>
                                <span class="assigned-user"><?= htmlspecialchars($task['assigned_to_fullname']) ?></span>
                                <?php else: ?>
                                <span class="no-assigned">Sin asignar</span>
                                <?php endif; ?>
                            </td>
                            <td class="td-due-date">
                                <?php if ($task['due_date']): ?>
                                <div class="due-date-info">
                                    <div class="due-date-text">
                                        <i class="fas fa-calendar"></i>
                                        <?= date('d/m/Y', strtotime($task['due_date'])) ?>
                                    </div>
                                </div>
                                <?php else: ?>
                                <span class="no-due-date">Sin fecha</span>
                                <?php endif; ?>
                            </td>
                            <td class="td-status">
                                <?php if ($canEditTask): ?>
                                <select class="status-select" onchange="updateTaskStatus(<?= $task['task_id'] ?>, this.value, event)">
                                    <option value="pending" <?= ($status==='pending')?'selected':'' ?>>Pendiente</option>
                                    <option value="in_progress" <?= ($status==='in_progress')?'selected':'' ?>>En Progreso</option>
                                    <option value="completed" <?= ($status==='completed')?'selected':'' ?>>Completada</option>
                                </select>
                                <?php else: ?>
                                <span class="status-badge status-<?= $status ?>">
                                    <?php 
                                    switch($status) {
                                        case 'in_progress': echo 'En Progreso'; break;
                                        case 'completed': echo 'Completada'; break;
                                        default: echo 'Pendiente'; break;
                                    }
                                    ?>
                                </span>
                                <?php endif; ?>
                            </td>
                            <td class="td-actions">
                                <div class="actions-group">
                                    <a href="<?= APP_URL ?>?route=clan_member/task-details&task_id=<?= $task['task_id'] ?>" class="btn-action-table" title="Ver detalles">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginación -->
        <div class="pagination-container">
            <div class="pagination-info">
                Mostrando <span id="showing-start">1</span> - <span id="showing-end"><?= min(5, count($tasks)) ?></span> de <span id="showing-total"><?= count($tasks) ?></span> tareas
            </div>
            <div class="pagination-buttons" id="paginationButtons">
                <!-- Los botones se generarán con JavaScript -->
            </div>
        </div>
    <?php else: ?>
        <!-- Estado vacío - No hay tareas -->
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-tasks"></i>
            </div>
            <h3>No hay tareas en este proyecto</h3>
            <p>Comienza creando tu primera tarea para este proyecto.</p>
            <button class="btn-nueva-tarea" onclick="openCreateTaskModal()">
                <i class="fas fa-plus"></i>
                Crear Primera Tarea
            </button>
        </div>
    <?php endif; ?>
</div>

<!-- Modal crear tarea - Diseño Monday.com -->
<div id="createTaskModal" class="modal-overlay" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <div style="display: flex; align-items: center; gap: 12px;">
                <img src="<?php echo Utils::asset('assets/img/Task_Loader.gif'); ?>" alt="Loading" style="width: 96px; height: 96px;">
                <h3>Nueva Tarea</h3>
            </div>
            <button type="button" class="modal-close" onclick="closeCreateTaskModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form id="createTaskForm" class="modal-body">
            <input type="hidden" name="project_id" value="<?php echo (int)($project['project_id'] ?? 0); ?>" />
            
            <div class="form-group">
                <label for="task_name">Título de la Tarea *</label>
                <input type="text" id="task_name" name="task_name" class="form-control" placeholder="Escribe un título claro y conciso" required>
            </div>
            
            <div class="form-group">
                <label for="task_description">Descripción</label>
                <textarea id="task_description" name="description" class="form-control" rows="3" placeholder="Contexto, entregables y consideraciones"></textarea>
            </div>
            
            <div class="form-group">
                <label for="task_priority">Prioridad</label>
                <select id="task_priority" name="priority" class="form-control">
                    <option value="low">Baja</option>
                    <option value="medium" selected>Media</option>
                    <option value="high">Alta</option>
                    <option value="urgent">Urgente</option>
                </select>
            </div>
            
            <div class="form-group" id="dueDateGroupMember">
                <label for="task_due_date">Fecha Límite</label>
                <input type="date" id="task_due_date" name="due_date" class="form-control">
            </div>
            
            <!-- Campos de Recurrencia -->
            <div class="form-group">
                <label class="checkbox-label-modern">
                    <input type="checkbox" id="isRecurrentMember" name="is_recurrent" onchange="toggleRecurrenceFieldsMember()">
                    <span class="checkbox-text">Tarea Recurrente</span>
                </label>
                <small class="field-help">Marca si esta tarea debe repetirse automáticamente</small>
            </div>
            
            <div id="recurrenceFieldsMember" class="recurrence-fields" style="display: none;">
                <div class="form-group">
                    <label for="recurrence_type">Tipo de Recurrencia</label>
                    <select id="recurrenceTypeMember" name="recurrence_type" class="form-control">
                        <option value="">Seleccionar...</option>
                        <option value="daily">Diaria</option>
                        <option value="weekly">Semanal</option>
                        <option value="monthly">Mensual</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="recurrence_start">Fecha de Inicio</label>
                    <input type="date" id="recurrenceStartMember" name="recurrence_start_date" class="form-control">
                    <small class="field-help">Primera fecha en que se ejecutará la tarea</small>
                </div>
                
                <div class="form-group">
                    <label for="recurrence_end">Fecha de Vigencia</label>
                    <input type="date" id="recurrenceEndMember" name="recurrence_end_date" class="form-control">
                    <small class="field-help">Hasta cuándo se generarán las repeticiones (opcional)</small>
                </div>
            </div>
            
            <div id="createTaskErrors" class="form-errors" style="display:none;"></div>
        </form>
        
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closeCreateTaskModal()">
                <i class="fas fa-times"></i>
                Cancelar
            </button>
            <button type="button" id="createTaskSubmitBtn" class="btn-primary" onclick="submitCreateTaskForm()">
                <i class="fas fa-plus"></i>
                Crear Tarea
            </button>
        </div>
    </div>
</div>

<script>
// Variables globales de paginación
let currentPage = 1;
let itemsPerPage = 5;
let allTasksArray = [];

// Función debounce para búsqueda
let searchTimeout;
function debounceSearch() {
    clearTimeout(searchTimeout);
    searchTimeout = setTimeout(() => {
        currentPage = 1; // Resetear a la primera página al buscar
        filterTasks();
    }, 300);
}

function openCreateTaskModal() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        
        // Establecer fecha mínima como hoy
        const dueDateField = document.getElementById('task_due_date');
        if (dueDateField) {
            dueDateField.min = new Date().toISOString().split('T')[0];
        }
    }
}

function closeCreateTaskModal() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
        // Limpiar formulario
        const form = document.getElementById('createTaskForm');
        if (form) form.reset();
        // Resetear campos de recurrencia
        const recurrenceFields = document.getElementById('recurrenceFieldsMember');
        if (recurrenceFields) recurrenceFields.style.display = 'none';
        const dueDateGroup = document.getElementById('dueDateGroupMember');
        if (dueDateGroup) dueDateGroup.style.display = 'block';
        // Limpiar errores
        const errorBox = document.getElementById('createTaskErrors');
        if (errorBox) errorBox.style.display = 'none';
    }
}

// Función para mostrar/ocultar campos de recurrencia
function toggleRecurrenceFieldsMember() {
    const checkbox = document.getElementById('isRecurrentMember');
    const fields = document.getElementById('recurrenceFieldsMember');
    const dueDateField = document.getElementById('task_due_date');
    const dueDateGroup = document.getElementById('dueDateGroupMember');
    
    if (checkbox.checked) {
        fields.style.display = 'block';
        if (dueDateGroup) {
            dueDateGroup.style.display = 'none';
        }
        if (dueDateField) {
            dueDateField.required = false;
            dueDateField.removeAttribute('required');
            dueDateField.value = '';
        }
        document.getElementById('recurrenceTypeMember').required = true;
        document.getElementById('recurrenceStartMember').required = true;
    } else {
        fields.style.display = 'none';
        if (dueDateGroup) {
            dueDateGroup.style.display = 'block';
        }
        if (dueDateField) {
            dueDateField.required = false;
        }
        document.getElementById('recurrenceTypeMember').required = false;
        document.getElementById('recurrenceStartMember').required = false;
        document.getElementById('recurrenceTypeMember').value = '';
        document.getElementById('recurrenceStartMember').value = '';
        document.getElementById('recurrenceEndMember').value = '';
    }
}

// Función para enviar el formulario de crear tarea
function submitCreateTaskForm() {
    const form = document.getElementById('createTaskForm');
    const title = form.querySelector('input[name="task_name"]').value.trim();
    const errorBox = document.getElementById('createTaskErrors');
    const submitBtn = document.getElementById('createTaskSubmitBtn');
    
    if(!title){
        errorBox.style.display = 'block';
        errorBox.textContent = 'El título es requerido.';
        return;
    }
    
    errorBox.style.display = 'none';
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando...';
    
    const fd = new FormData(form);
    const projectId = form.querySelector('input[name="project_id"]').value;
    
    fetch('?route=clan_member/create-project-task', { method:'POST', body: fd, credentials:'same-origin' })
        .then(async r=>{ 
            const t = await r.text(); 
            try{ 
                return JSON.parse(t); 
            } catch(e){ 
                console.error(t); 
                return {success:false,message:'Respuesta inválida'}; 
            } 
        })
        .then(d=>{ 
            if(!d.success){ 
                errorBox.style.display='block'; 
                errorBox.textContent=d.message||'Error al crear la tarea'; 
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-plus"></i> Crear Tarea';
                return; 
            }
            
            // Redirigir inmediatamente
            window.location.href = '?route=clan_member/project-tasks&project_id=' + projectId;
        })
        .catch(()=>{ 
            errorBox.style.display='block'; 
            errorBox.textContent='Error de red'; 
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-plus"></i> Crear Tarea';
        });
}

// Cerrar modal con ESC
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const modal = document.getElementById('createTaskModal');
        if (modal && modal.style.display === 'flex') {
            closeCreateTaskModal();
        }
    }
});

// Cerrar modal al hacer clic fuera
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                closeCreateTaskModal();
            }
        });
    }
});

function filterTasks() {
    const statusFilter = document.getElementById('status_filter').value;
    const priorityFilter = document.getElementById('priority_filter').value;
    const searchText = document.getElementById('searchInput').value.toLowerCase();
    const taskRows = document.querySelectorAll('.task-row');
    
    // Filtrar tareas
    let visibleTasks = [];
    taskRows.forEach(row => {
        const status = row.dataset.status;
        const priority = row.dataset.priority;
        const taskName = row.querySelector('.task-title')?.textContent.toLowerCase() || '';
        const taskDesc = row.querySelector('.task-description')?.textContent.toLowerCase() || '';
        
        const statusMatch = !statusFilter || status === statusFilter;
        const priorityMatch = !priorityFilter || priority === priorityFilter;
        const searchMatch = !searchText || taskName.includes(searchText) || taskDesc.includes(searchText);
        
        if (statusMatch && priorityMatch && searchMatch) {
            visibleTasks.push(row);
        }
    });
    
    // Guardar tareas visibles para paginación
    allTasksArray = visibleTasks;
    
    // Aplicar paginación
    applyPagination();
}

function applyPagination() {
    const totalTasks = allTasksArray.length;
    const totalPages = itemsPerPage === 'all' ? 1 : Math.ceil(totalTasks / itemsPerPage);
    
    // Asegurar que currentPage esté en rango válido
    if (currentPage > totalPages && totalPages > 0) {
        currentPage = totalPages;
    }
    if (currentPage < 1) {
        currentPage = 1;
    }
    
    // Ocultar todas las filas primero
    document.querySelectorAll('.task-row').forEach(row => {
        row.style.display = 'none';
    });
    
    // Mostrar solo las filas de la página actual
    if (itemsPerPage === 'all') {
        allTasksArray.forEach(row => {
            row.style.display = 'table-row';
        });
    } else {
        const start = (currentPage - 1) * itemsPerPage;
        const end = start + itemsPerPage;
        
        for (let i = start; i < end && i < allTasksArray.length; i++) {
            allTasksArray[i].style.display = 'table-row';
        }
    }
    
    // Actualizar info de paginación
    updatePaginationInfo(totalTasks, totalPages);
}

function updatePaginationInfo(totalTasks, totalPages) {
    const showingStart = totalTasks === 0 ? 0 : ((currentPage - 1) * itemsPerPage) + 1;
    const showingEnd = itemsPerPage === 'all' ? totalTasks : Math.min(currentPage * itemsPerPage, totalTasks);
    
    document.getElementById('showing-start').textContent = showingStart;
    document.getElementById('showing-end').textContent = showingEnd;
    document.getElementById('showing-total').textContent = totalTasks;
    
    // Generar botones de paginación
    const paginationButtons = document.getElementById('paginationButtons');
    paginationButtons.innerHTML = '';
    
    if (itemsPerPage !== 'all' && totalPages > 1) {
        // Botón anterior
        const prevBtn = document.createElement('button');
        prevBtn.className = 'pagination-btn';
        prevBtn.innerHTML = '<i class="fas fa-chevron-left"></i>';
        prevBtn.disabled = currentPage === 1;
        prevBtn.onclick = () => {
            if (currentPage > 1) {
                currentPage--;
                applyPagination();
            }
        };
        paginationButtons.appendChild(prevBtn);
        
        // Botones de páginas
        for (let i = 1; i <= totalPages; i++) {
            // Mostrar solo algunas páginas alrededor de la actual
            if (i === 1 || i === totalPages || (i >= currentPage - 2 && i <= currentPage + 2)) {
                const pageBtn = document.createElement('button');
                pageBtn.className = 'pagination-btn' + (i === currentPage ? ' active' : '');
                pageBtn.textContent = i;
                pageBtn.onclick = () => {
                    currentPage = i;
                    applyPagination();
                };
                paginationButtons.appendChild(pageBtn);
            } else if (i === currentPage - 3 || i === currentPage + 3) {
                const dots = document.createElement('span');
                dots.textContent = '...';
                dots.style.padding = '0 8px';
                paginationButtons.appendChild(dots);
            }
        }
        
        // Botón siguiente
        const nextBtn = document.createElement('button');
        nextBtn.className = 'pagination-btn';
        nextBtn.innerHTML = '<i class="fas fa-chevron-right"></i>';
        nextBtn.disabled = currentPage === totalPages;
        nextBtn.onclick = () => {
            if (currentPage < totalPages) {
                currentPage++;
                applyPagination();
            }
        };
        paginationButtons.appendChild(nextBtn);
    }
}

function changePerPage() {
    const perPageValue = document.getElementById('perPage').value;
    itemsPerPage = perPageValue === 'all' ? 'all' : parseInt(perPageValue);
    currentPage = 1;
    filterTasks();
}

function resetFilters() {
    document.getElementById('status_filter').value = '';
    document.getElementById('priority_filter').value = '';
    document.getElementById('searchInput').value = '';
    document.getElementById('perPage').value = '5';
    itemsPerPage = 5;
    currentPage = 1;
    filterTasks();
}

// Inicializar paginación al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    filterTasks();
});

function updateStats() {
    // Las estadísticas deben reflejar el total del proyecto, NO los filtros
    // No hacer nada aquí ya que los valores iniciales de PHP son los correctos
    // Esta función se mantiene por compatibilidad pero no modifica los cards
}

function updateTaskStatus(taskId, newStatus, event) {
    // Guardar referencia al select y la fila antes de hacer la petición
    const selectElement = event ? event.target : document.querySelector(`select[onchange*="${taskId}"]`);
    const taskRow = selectElement.closest('.task-row');
    const originalStatus = taskRow.dataset.status;
    
    // Deshabilitar el select mientras se procesa
    selectElement.disabled = true;
    
    const fd = new FormData();
    fd.append('task_id', taskId);
    
    // Determinar qué endpoint usar según el estado
    let endpoint;
    if (newStatus === 'completed' || newStatus === 'pending') {
        fd.append('is_completed', newStatus === 'completed' ? 'true' : 'false');
        endpoint = '?route=clan_member/toggle-task-status';
    } else {
        fd.append('status', newStatus);
        endpoint = '?route=clan_member/update-task';
    }
    
    // Hacer la petición
    fetch(endpoint, { 
        method: 'POST', 
        body: fd, 
        credentials: 'same-origin' 
    })
    .then(async response => { 
        const text = await response.text(); 
        try { 
            return JSON.parse(text); 
        } catch(e) { 
            console.error('Error parsing response:', text); 
            return { success: false, message: 'Respuesta inválida del servidor' }; 
        } 
    })
    .then(data => { 
        if (!data.success) { 
            alert(data.message || 'Error al actualizar el estado'); 
            selectElement.value = originalStatus;
        } else {
            // Mostrar animación de completado si cambió a completado
            if (newStatus === 'completed' && originalStatus !== 'completed') {
                showCompletionAnimation();
            }
            
            // Actualizar el DOM localmente
            taskRow.dataset.status = newStatus;
            
            // Actualizar clases visuales
            if (newStatus === 'completed') {
                taskRow.classList.add('completed');
            } else {
                taskRow.classList.remove('completed');
            }
            
            // Mostrar feedback visual temporal
            taskRow.style.backgroundColor = '#d4edda';
            setTimeout(() => {
                taskRow.style.backgroundColor = '';
            }, 1000);
        }
    })
    .catch(error => {
        console.error('Error en la petición:', error);
        alert('Error de conexión al actualizar el estado');
        selectElement.value = originalStatus;
    })
    .finally(() => {
        // Siempre habilitar el select de nuevo
        selectElement.disabled = false;
    });
}

function showCompletionAnimation() {
    console.log('🎉 Mostrando animación de celebración');
    
    // Remover overlay anterior si existe
    const existingOverlay = document.getElementById('completion-animation-overlay');
    if (existingOverlay) {
        existingOverlay.remove();
    }
    
    // Crear overlay
    const overlay = document.createElement('div');
    overlay.id = 'completion-animation-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.4);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 99999;
        opacity: 0;
        transition: opacity 0.3s ease;
    `;
    
    // Crear el elemento lottie
    const lottiePlayer = document.createElement('dotlottie-wc');
    lottiePlayer.setAttribute('src', 'https://lottie.host/3fb875a8-1324-4e18-bb92-116a8e77a2d7/E2KxUaNO1B.lottie');
    lottiePlayer.setAttribute('autoplay', '');
    lottiePlayer.style.cssText = 'width: 350px; height: 350px;';
    
    overlay.appendChild(lottiePlayer);
    document.body.appendChild(overlay);
    
    // Mostrar con animación
    requestAnimationFrame(() => {
        overlay.style.opacity = '1';
    });
    
    // Ocultar después de 3 segundos
    setTimeout(() => {
        overlay.style.opacity = '0';
        setTimeout(() => {
            overlay.remove();
        }, 300);
    }, 3000);
    
    console.log('🎉 Animación creada y mostrada');
}
</script>

<?php
$content = ob_get_clean();
$additionalCSS = [];
require_once __DIR__ . '/../layout.php';
?>
