<?php
// Iniciar buffer de salida
ob_start();

// Verificar que tenemos los datos necesarios
if (!isset($project) || !isset($tasks)) {
    echo '<div style="padding: 20px; text-align: center;">Error: Datos del proyecto no disponibles</div>';
    $content = ob_get_clean();
    require_once __DIR__ . '/../layout.php';
    return;
}

$projectName = htmlspecialchars($project['project_name'] ?? 'Proyecto');
$clanName = htmlspecialchars($clan['clan_name'] ?? 'Desarrollo');
?>

<!-- Cargar CSS Monday Theme -->
<link rel="stylesheet" href="<?= APP_URL ?>/assets/css/monday-theme.css">
<script src="https://unpkg.com/@lottiefiles/dotlottie-wc@0.8.5/dist/dotlottie-wc.js" type="module"></script>

<style>
/* ===== ESTILOS HOMOLOGADOS CON TASKS.PHP ===== */

/* Sección principal */
.all-tasks-section {
    background: var(--monday-surface, #ffffff);
    border-radius: 8px;
    padding: 24px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
    border: 1px solid var(--monday-border-color, #d0d4e4);
}

.section-header {
    margin-bottom: 20px;
}

.section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--monday-text-color, #323338);
    margin: 0;
}

/* Tarjetas de estadísticas */
.stats-cards-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}

.stat-card {
    background: white;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    border: 1px solid var(--monday-border-color, #d0d4e4);
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12);
}

.stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    color: white;
    background: #3b82f6;
    flex-shrink: 0;
}

.stat-icon.total { background: #3b82f6; }
.stat-icon.completed { background: #10b981; }
.stat-icon.in-progress { background: #3b82f6; }
.stat-icon.pending { background: #f59e0b; }

.stat-content { flex: 1; }

.stat-value {
    font-size: 32px;
    font-weight: 700;
    color: #1f2937;
    line-height: 1;
    margin-bottom: 4px;
}

.stat-label {
    font-size: 12px;
    font-weight: 600;
    color: #6b7280;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Header con botón */
.section-header-with-action {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 16px;
}

.header-content-wrapper {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    width: 100%;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 16px;
}

.header-content-left {
    flex: 1;
    min-width: 200px;
}

.header-content-right {
    display: flex;
    align-items: center;
    gap: 16px;
}

.project-title-header {
    font-size: 24px;
    font-weight: 700;
    color: var(--monday-text-color, #323338);
    margin: 0 0 8px 0;
    display: flex;
    align-items: center;
    gap: 8px;
}

.project-icon-header {
    color: #0073ea;
    font-size: 20px;
}

.project-description-box {
    margin-top: 12px;
    padding: 12px 16px;
    background: #f8f9fa;
    border-radius: 8px;
    border-left: 4px solid #0073ea;
}

.project-description-box p {
    font-size: 14px;
    color: #676879;
    line-height: 1.6;
    margin: 0;
}

.header-actions-wrapper {
    display: flex;
    align-items: center;
    gap: 16px;
}

.lottie-animation-container {
    display: flex;
    justify-content: center;
    align-items: center;
}

.btn-header-action {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    font-size: 14px;
    cursor: pointer;
    transition: all 0.2s;
}

.btn-proyectos {
    background: #f1f5f9;
    color: #475569;
    border: 1px solid #e2e8f0;
}

.btn-proyectos:hover {
    background: #e2e8f0;
    border-color: #cbd5e1;
    transform: translateY(-1px);
}

.btn-nueva-tarea {
    background: linear-gradient(135deg, #059669 0%, #10b981 100%);
    color: white !important;
    border: none;
    box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
}

.btn-nueva-tarea:hover {
    background: linear-gradient(135deg, #047857 0%, #059669 100%);
    transform: translateY(-2px);
    color: white !important;
    box-shadow: 0 8px 20px rgba(16, 185, 129, 0.4);
}

/* Filtros estilo Monday */
.filters-container {
    margin-bottom: 20px;
}

.filter-row {
    display: flex;
    flex-direction: row;
    flex-wrap: nowrap;
    gap: 16px;
    align-items: center;
    margin-bottom: 12px;
}

.filter-item {
    display: inline-flex;
    flex-direction: row;
    align-items: center;
    gap: 8px;
    white-space: nowrap;
}

.filter-item label {
    font-size: 13px;
    font-weight: 500;
    color: var(--monday-text-secondary, #676879);
}

.filter-item select {
    padding: 8px 12px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    color: var(--monday-text-color, #323338);
    background: white;
    min-width: 140px;
    cursor: pointer;
}

.filter-item select:hover,
.filter-item select:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
}

.btn-reset-filters {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 14px;
    background: transparent;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    font-weight: 500;
    color: var(--monday-text-secondary, #676879);
    cursor: pointer;
}

.btn-reset-filters:hover {
    background: #f5f6f8;
    border-color: #c4c4c4;
}

/* Búsqueda */
.search-row {
    display: block;
    max-width: 400px;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: var(--monday-text-secondary, #676879);
    font-size: 14px;
}

.search-input {
    width: 100%;
    padding: 8px 12px 8px 36px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 13px;
    color: var(--monday-text-color, #323338);
}

.search-input:hover,
.search-input:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
    box-shadow: 0 0 0 2px rgba(0, 115, 234, 0.1);
}

/* ===== TABLA MONDAY - ESTILOS IDÉNTICOS A TASKS.PHP ===== */
.tasks-table-container {
    overflow-x: auto;
    margin-top: 16px;
}

.tasks-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.tasks-table thead {
    background: #f5f6f8;
    border-bottom: 1px solid var(--monday-border-color, #d0d4e4);
}

.tasks-table th {
    padding: 12px 16px;
    text-align: left;
    font-weight: 600;
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: var(--monday-text-secondary, #676879);
    white-space: nowrap;
}

.tasks-table tbody tr {
    border-bottom: 1px solid #eaebef;
    transition: background-color 0.15s;
}

.tasks-table tbody tr:hover {
    background: rgba(0, 115, 234, 0.04);
}

.tasks-table tbody tr.completed {
    background: rgba(0, 200, 117, 0.05);
}

.tasks-table td {
    padding: 14px 16px;
    color: var(--monday-text-color, #323338);
    vertical-align: middle;
}

/* Estilos para checkbox */
.th-checkbox, .td-checkbox {
    width: 60px !important;
    min-width: 60px !important;
    max-width: 60px !important;
    text-align: center;
    font-size: 12px;
    padding: 12px 10px !important;
    font-weight: 600;
    color: #6b7280;
}

.td-checkbox input[type="checkbox"] {
    width: 18px;
    height: 18px;
    cursor: pointer;
    accent-color: #00c875;
}

/* Columnas específicas */
.th-priority, .td-priority {
    width: 100px !important;
    min-width: 100px !important;
    max-width: 100px !important;
    text-align: center;
    white-space: nowrap;
    overflow: visible !important;
}

.th-task, .td-task {
    width: 25%;
    min-width: 200px;
}

.th-project, .td-project {
    width: 120px;
    min-width: 100px;
    max-width: 140px;
}

.th-assigned, .td-assigned {
    width: 12%;
    min-width: 100px;
}

.th-due-date, .td-due-date {
    width: 100px;
    min-width: 90px;
    max-width: 110px;
}

.th-status, .td-status {
    width: 110px !important;
    min-width: 110px !important;
    max-width: 110px !important;
    text-align: center;
    white-space: nowrap;
    overflow: visible !important;
}

.th-progress, .td-progress {
    width: 80px;
    min-width: 80px;
    max-width: 80px;
    text-align: center;
    padding: 0.875rem 1.25rem;
    border-right: 2px solid #e5e7eb;
    box-sizing: border-box;
}

.th-actions, .td-actions {
    width: 220px;
    min-width: 220px;
    max-width: 220px;
    text-align: center;
    padding: 0.875rem 1.25rem;
    border-right: none;
    box-sizing: border-box;
}

.th-select, .td-select {
    width: 60px !important;
    min-width: 60px !important;
    text-align: center;
}

/* Badge de prioridad - ESTILOS EXACTOS DE TASKS.PHP */
.priority-badge {
    display: inline-block;
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.priority-badge.priority-critical,
.priority-badge.critical,
.priority-badge.urgent,
.priority-badge.priority-urgent {
    background: #dc2626;
    color: white;
}

.priority-badge.priority-high,
.priority-badge.high {
    background: #ea580c;
    color: white;
}

.priority-badge.priority-medium,
.priority-badge.medium {
    background: #d97706;
    color: white;
}

.priority-badge.priority-low,
.priority-badge.low {
    background: #059669;
    color: white;
}

/* Info de tarea - ESTILOS IDÉNTICOS A TASKS.PHP */
.task-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.task-title {
    font-weight: 600;
    color: #1f2937;
    line-height: 1.3;
}

.task-description {
    color: #6b7280;
    font-size: 0.8rem;
    line-height: 1.4;
}

.task-name-table {
    display: flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: #1f2937;
}

.task-description-table {
    color: #6b7280;
    font-size: 0.8rem;
    line-height: 1.4;
    margin-top: 4px;
}

.assigned-user,
.assigned-users {
    font-weight: 500;
    color: #0073ea;
}

.no-assigned {
    color: #9ca3af;
    font-style: italic;
    font-size: 0.8rem;
}

.project-name,
.project-badge {
    display: inline-block;
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    background: #dbeafe;
    color: #1e40af;
}

/* Fecha de vencimiento */
.due-date-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.due-date-text {
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 4px;
    color: var(--monday-text-color, #323338);
    font-size: 13px;
}

.due-date-text i {
    font-size: 11px;
    color: var(--monday-text-secondary, #676879);
}

.no-due-date {
    color: var(--monday-text-secondary, #676879);
    font-size: 12px;
    font-style: italic;
}

/* Badge de fecha de vencimiento - ESTILOS EXACTOS DE TASKS.PHP */
.due-date-badge {
    display: inline-block;
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.due-date-badge.overdue {
    background: #fee2e2;
    color: #991b1b;
}

.due-date-badge.today {
    background: #fef3c7;
    color: #92400e;
}

.due-date-badge.week1 {
    background: #dbeafe;
    color: #1e40af;
}

.due-date-badge.week2 {
    background: #d1fae5;
    color: #065f46;
}

/* Badge de estado - ESTILOS EXACTOS DE TASKS.PHP */
.status-badge {
    display: inline-block;
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge.status-pending,
.status-badge.pending {
    background: #fef3c7;
    color: #92400e;
}

.status-badge.status-in_progress,
.status-badge.in_progress {
    background: #dbeafe;
    color: #1e40af;
}

.status-badge.status-completed,
.status-badge.completed {
    background: #d1fae5;
    color: #065f46;
}

/* Barra de progreso - ESTILOS EXACTOS DE TASKS.PHP */
.progress-container-table {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 4px;
    padding: 4px;
    width: 100%;
    box-sizing: border-box;
}

.progress-bar-table {
    width: 100%;
    height: 6px;
    background: #e5e7eb;
    border-radius: 3px;
    overflow: hidden;
    min-width: 60px;
    max-width: 80px;
    box-shadow: inset 0 1px 3px rgba(0,0,0,0.1);
}

.progress-fill-table {
    height: 100%;
    background: #00c875;
    border-radius: 3px;
    transition: width 0.3s ease;
}

.progress-text-table {
    font-size: 10px;
    font-weight: 600;
    color: #6b7280;
    text-align: center;
    white-space: nowrap;
}

/* Select de estado */
.status-select {
    padding: 6px 10px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    font-size: 12px;
    background: white;
    color: var(--monday-text-color, #323338);
    font-weight: 600;
    cursor: pointer;
    min-width: 120px;
}

.status-select:hover,
.status-select:focus {
    border-color: var(--monday-primary, #0073ea);
    outline: none;
}

/* Botones de acción - ESTILOS IDÉNTICOS A TASKS.PHP */
.actions-group {
    display: flex;
    gap: 6px;
    justify-content: center;
    padding: 0 4px;
    align-items: center;
    flex-wrap: nowrap;
}

.btn-action-table {
    display: inline-flex !important;
    align-items: center !important;
    justify-content: center !important;
    width: 32px !important;
    height: 32px !important;
    border: none !important;
    border-radius: 6px !important;
    font-size: 13px !important;
    cursor: pointer !important;
    text-decoration: none !important;
    background: #669AE1 !important;
    color: white !important;
    transition: all 0.2s !important;
}

.btn-action-table:hover {
    background: #4a7bc8 !important;
    transform: translateY(-1px) !important;
}

.btn-action-table i,
.btn-action-table .fas,
.btn-action-table .fa {
    color: white !important;
}

/* Paginación */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 16px 0;
    margin-top: 16px;
    border-top: 1px solid var(--monday-border-color, #d0d4e4);
}

.pagination-info {
    font-size: 13px;
    color: var(--monday-text-secondary, #676879);
}

.pagination-buttons {
    display: flex;
    gap: 4px;
    align-items: center;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 32px;
    height: 32px;
    padding: 0 10px;
    border: 1px solid var(--monday-border-color, #d0d4e4);
    border-radius: 6px;
    background: white;
    color: var(--monday-text-color, #323338);
    font-size: 13px;
    cursor: pointer;
    transition: all 0.2s;
}

.pagination-btn:hover:not(:disabled):not(.active) {
    background: #0073ea;
    border-color: #0073ea;
    color: white;
}

.pagination-btn.active {
    background: #0073ea;
    border-color: #0073ea;
    color: white;
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Estado vacío */
.empty-state {
    text-align: center;
    padding: 60px 20px;
}

.empty-state .empty-icon {
    font-size: 48px;
    color: var(--monday-text-secondary, #676879);
    margin-bottom: 16px;
}

.empty-state h3 {
    font-size: 18px;
    color: var(--monday-text-color, #323338);
    margin: 0 0 8px 0;
}

.empty-state p {
    color: var(--monday-text-secondary, #676879);
    margin: 0 0 20px 0;
}

/* ===== ESTILOS DEL MODAL DE CREAR TAREA (EXACTOS DE TASKS.PHP) ===== */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(15, 23, 42, 0.6);
    backdrop-filter: blur(10px);
    -webkit-backdrop-filter: blur(10px);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

@keyframes fadeIn {
    from {
        opacity: 0;
    }
    to {
        opacity: 1;
    }
}

.modal-content {
    background: rgba(252, 252, 253, 0.9);
    backdrop-filter: blur(40px) saturate(105%);
    -webkit-backdrop-filter: blur(40px) saturate(105%);
    border-radius: 20px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow: hidden;
    box-shadow: 
        0 8px 32px rgba(16, 185, 129, 0.04),
        0 0 0 1px rgba(255, 255, 255, 0.4) inset,
        0 1px 2px rgba(0, 0, 0, 0.05);
    border: 1px solid rgba(236, 253, 245, 0.08);
    animation: scaleIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    display: flex;
    flex-direction: column;
}

.modal-content::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 100%;
    background: linear-gradient(
        135deg,
        rgba(236, 253, 245, 0.02) 0%,
        rgba(209, 250, 229, 0.01) 50%,
        rgba(252, 252, 253, 0.01) 100%
    );
    pointer-events: none;
    border-radius: 20px;
}

@keyframes scaleIn {
    from {
        opacity: 0;
        transform: scale(0.95) translateY(20px);
    }
    to {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

.modal-header {
    padding: 24px 28px;
    border-bottom: 1px solid rgba(236, 253, 245, 0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(to bottom, rgba(252, 252, 253, 0.6), rgba(236, 253, 245, 0.08));
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    position: relative;
    z-index: 1;
    flex-shrink: 0;
}

.modal-header > div {
    display: flex;
    align-items: center;
    gap: 12px;
    flex: 1;
    min-width: 0;
}

.modal-header > div img {
    flex-shrink: 0;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
    background: linear-gradient(135deg, #1e293b 0%, #475569 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    letter-spacing: -0.02em;
}

.modal-close {
    background: rgba(239, 68, 68, 0.1);
    border: 1px solid rgba(239, 68, 68, 0.2);
    backdrop-filter: blur(10px);
    -webkit-backdrop-filter: blur(10px);
    font-size: 1.25rem;
    color: #ef4444;
    cursor: pointer;
    padding: 0;
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 10px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.modal-close:hover {
    background: rgba(239, 68, 68, 0.2);
    border-color: rgba(239, 68, 68, 0.4);
    transform: scale(1.05);
    box-shadow: 0 4px 12px rgba(239, 68, 68, 0.2);
}

.modal-body {
    padding: 28px;
    overflow-y: auto;
    position: relative;
    z-index: 1;
    background: rgba(252, 252, 253, 0.4);
    flex: 1;
    min-height: 0;
}

.modal-body .form-group {
    margin-bottom: 20px;
}

.modal-body .form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #1e293b;
    font-size: 0.9rem;
    letter-spacing: -0.01em;
}

.modal-body .form-control {
    width: 100%;
    padding: 12px 16px;
    background: rgba(255, 255, 255, 0.8);
    backdrop-filter: blur(10px);
    -webkit-backdrop-filter: blur(10px);
    border: 1px solid rgba(236, 253, 245, 0.15);
    border-radius: 12px;
    font-size: 0.95rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
    color: #1e293b;
    box-sizing: border-box;
}

.modal-body .form-control::placeholder {
    color: rgba(148, 163, 184, 0.6);
}

.modal-body .form-control:focus {
    outline: none;
    background: rgba(255, 255, 255, 0.9);
    border-color: rgba(16, 185, 129, 0.15);
    box-shadow: 
        0 0 0 3px rgba(16, 185, 129, 0.04),
        0 4px 12px rgba(16, 185, 129, 0.05);
    transform: translateY(-1px);
}

.modal-footer {
    padding: 20px 28px 24px;
    border-top: 1px solid rgba(236, 253, 245, 0.1);
    display: flex;
    gap: 12px;
    justify-content: flex-end;
    background: linear-gradient(to top, rgba(252, 252, 253, 0.6), rgba(236, 253, 245, 0.08));
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    position: relative;
    z-index: 1;
    flex-shrink: 0;
}

.modal-footer .btn-secondary {
    padding: 12px 24px;
    border: 1px solid rgba(226, 232, 240, 0.5);
    background: rgba(255, 255, 255, 0.7);
    backdrop-filter: blur(10px);
    -webkit-backdrop-filter: blur(10px);
    color: #475569;
    border-radius: 10px;
    cursor: pointer;
    font-size: 0.9rem;
    font-weight: 600;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.modal-footer .btn-secondary:hover {
    background: rgba(255, 255, 255, 0.9);
    border-color: rgba(59, 130, 246, 0.5);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
}

.modal-footer .btn-primary {
    padding: 12px 28px;
    border: none;
    background: linear-gradient(135deg, #059669 0%, #10b981 100%);
    color: white;
    border-radius: 10px;
    cursor: pointer;
    font-size: 0.9rem;
    font-weight: 600;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 
        0 4px 12px rgba(16, 185, 129, 0.3),
        0 0 0 1px rgba(255, 255, 255, 0.1) inset;
    position: relative;
    overflow: hidden;
}

.modal-footer .btn-primary::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
    transition: left 0.5s;
}

.modal-footer .btn-primary:hover::before {
    left: 100%;
}

.modal-footer .btn-primary:hover {
    background: linear-gradient(135deg, #047857 0%, #059669 100%);
    transform: translateY(-2px);
    box-shadow: 
        0 8px 20px rgba(16, 185, 129, 0.4),
        0 0 0 1px rgba(255, 255, 255, 0.2) inset;
}

.modal-footer .btn-primary:active {
    transform: translateY(0);
}

.modal-body::-webkit-scrollbar {
    width: 8px;
}

.modal-body::-webkit-scrollbar-track {
    background: rgba(0, 0, 0, 0.05);
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb {
    background: rgba(148, 163, 184, 0.3);
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb:hover {
    background: rgba(148, 163, 184, 0.5);
}

/* Estilos responsivos mejorados para el modal */
@media (max-width: 768px) {
    .modal-content {
        width: 100%;
        height: 100%;
        max-height: 100%;
        border-radius: 0;
        display: flex;
        flex-direction: column;
    }

    .modal-header {
        padding: 16px;
        flex-shrink: 0;
    }
    
    .modal-header > div img {
        width: 40px !important;
        height: 40px !important;
    }
    
    .modal-header h3 {
        font-size: 1.1rem;
    }
    
    .modal-body {
        padding: 16px;
        flex: 1;
        overflow-y: auto;
        -webkit-overflow-scrolling: touch; /* Smooth scrolling en iOS */
    }

    .modal-footer {
        padding: 16px;
        flex-shrink: 0; /* Evita que el footer se encoja */
        background: white; /* Asegura fondo sólido en móviles */
        border-top: 1px solid #e2e8f0;
    }
}

/* Estilos mejorados para la selección de miembros */
.members-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 10px;
    max-height: 250px;
    overflow-y: auto;
    padding: 4px;
}

.member-card {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 8px 12px;
    background: rgba(255, 255, 255, 0.6);
    border: 1px solid rgba(236, 253, 245, 0.3);
    border-radius: 10px;
    cursor: pointer;
    transition: all 0.2s ease;
    user-select: none;
    position: relative;
}

.member-card:hover {
    background: rgba(255, 255, 255, 0.9);
    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    border-color: rgba(16, 185, 129, 0.3);
}

.member-card.selected {
    background: #d1fae5;
    border-color: #10b981;
    box-shadow: 0 0 0 1px #10b981;
}

.member-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    color: #475569;
    font-size: 12px;
    flex-shrink: 0;
}

.member-name {
    font-size: 0.9rem;
    color: #1e293b;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    flex: 1;
}

.member-check {
    width: 18px;
    height: 18px;
    border: 2px solid #cbd5e1;
    border-radius: 4px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;
    flex-shrink: 0;
}

.member-card.selected .member-check {
    background: #10b981;
    border-color: #10b981;
}

.member-check::after {
    content: '✓';
    color: white;
    font-size: 12px;
    display: none;
}

.member-card.selected .member-check::after {
    display: block;
}

.member-search {
    margin-bottom: 10px;
    position: relative;
}

.member-search i {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: #94a3b8;
}

.member-search input {
    padding-left: 36px !important;
}

@media (max-width: 480px) {
    .members-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="all-tasks-section">
    <div class="header-content-wrapper">
        <div class="header-content-left">
            <h1 class="project-title-header">
                <i class="fas fa-folder project-icon-header"></i>
                <?= $projectName ?>
            </h1>
            <?php if (!empty($project['description'])): ?>
                <?php
                $fullDescription = htmlspecialchars($project['description']);
                $shortDescription = htmlspecialchars(mb_substr($project['description'], 0, 100));
                $displayDescription = (mb_strlen($project['description']) > 100) ? $shortDescription . '...' : $fullDescription;
                ?>
                <div class="project-description-box" title="<?= $fullDescription ?>">
                    <p><?= $displayDescription ?></p>
                </div>
            <?php endif; ?>
        </div>
        <div class="header-content-right">
            <div class="header-actions-wrapper">
                <a href="?route=clan_leader/projects" class="btn-header-action btn-proyectos">
                    <i class="fas fa-folder"></i>
                    Proyectos
                </a>
                <button type="button" class="btn-header-action btn-nueva-tarea" onclick="openCreateTaskModal(<?= $project['project_id'] ?>)">
                    <i class="fas fa-plus"></i>
                    Nueva Tarea
                </button>
            </div>
        </div>
    </div>
    
    <!-- Tarjetas de estadísticas -->
    <div class="stats-cards-container">
        <div class="stat-card">
            <div class="stat-icon total">
                <i class="fas fa-list"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-total"><?= count($tasks) ?></div>
                <div class="stat-label">Total Tareas</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon completed">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-completed"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'completed'; })) ?></div>
                <div class="stat-label">Completadas</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon in-progress">
                <i class="fas fa-spinner"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-in-progress"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'in_progress'; })) ?></div>
                <div class="stat-label">En Progreso</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon pending">
                <i class="fas fa-clock"></i>
            </div>
            <div class="stat-content">
                <div class="stat-value" id="stat-pending"><?= count(array_filter($tasks, function($t) { return ($t['status'] ?? 'pending') === 'pending'; })) ?></div>
                <div class="stat-label">Pendientes</div>
            </div>
        </div>
    </div>
    
    <!-- Animación Lottie -->
    <div class="lottie-animation-container" style="margin: 20px 0;">
        <dotlottie-wc src="https://lottie.host/46d0b721-b84d-49de-be3a-59f0c95d2a4f/4EVkAXm90o.lottie" style="width: 160px;height: 160px" autoplay loop></dotlottie-wc>
    </div>
    
    <?php if (isset($tasks) && !empty($tasks)): ?>
        <!-- Filtros y búsqueda -->
        <div class="filters-container">
            <div class="filter-row">
                <div class="filter-item">
                    <label for="status_filter">Estado:</label>
                    <select id="status_filter" onchange="filterTasks()">
                        <option value="">Todos</option>
                        <option value="pending">Pendiente</option>
                        <option value="in_progress">En Progreso</option>
                        <option value="completed">Completado</option>
                    </select>
                </div>
                
                <div class="filter-item">
                    <label for="perPage">Mostrar:</label>
                    <select id="perPage" onchange="changePerPage()">
                        <option value="5" selected>5 por página</option>
                        <option value="10">10 por página</option>
                        <option value="25">25 por página</option>
                        <option value="50">50 por página</option>
                        <option value="all">Todos</option>
                    </select>
                </div>
                
                <button type="button" class="btn-reset-filters" onclick="resetFilters()">
                    <i class="fas fa-undo"></i>
                    Limpiar
                </button>
            </div>
            
            <div class="search-row">
                <div class="search-input-wrapper">
                    <i class="fas fa-search search-icon"></i>
                    <input type="text" 
                           id="searchInput"
                           placeholder="Buscar tareas..."
                           class="search-input"
                           oninput="debounceSearch()">
                </div>
            </div>
        </div>
        
        <!-- Tabla de Tareas -->
        <div class="tasks-table-container">
            <table class="tasks-table">
                <thead>
                    <tr>
                        <th class="th-checkbox" style="width: 50px;">
                            <i class="fas fa-check"></i>
                        </th>
                        <th class="th-priority">Prioridad</th>
                        <th class="th-task">Tarea</th>
                        <th class="th-project">Proyecto</th>
                        <th class="th-assigned">Asignado</th>
                        <th class="th-due-date">Fecha Límite</th>
                        <th class="th-status">Estado</th>
                        <th class="th-progress">Progreso</th>
                        <th class="th-actions">Acciones</th>
                        <th class="th-select" style="width: 60px;">Selección</th>
                    </tr>
                </thead>
                <tbody id="tasksTableBody">
                    <?php foreach ($tasks as $task): ?>
                        <?php 
                        $userId = (int)($user['user_id'] ?? 0);
                        $assignedIds = array_filter(explode(',', (string)($task['all_assigned_user_ids'] ?? '')));
                        $canEditTask = in_array((string)$userId, $assignedIds, true) || (int)($task['assigned_to_user_id'] ?? 0) === $userId || (int)($task['created_by_user_id'] ?? 0) === $userId;
                        
                        $status = $task['status'] ?? 'pending';
                        $priority = $task['priority'] ?? 'medium';
                        
                        // Prioridad texto
                        $priorityText = $priority === 'critical' ? 'Urgente' : ($priority === 'urgent' ? 'Urgente' : ($priority === 'high' ? 'Alta' : ($priority === 'low' ? 'Baja' : 'Media')));
                        ?>
                        <tr class="task-row priority-<?= $priority ?> <?= ($status === 'completed') ? 'completed' : '' ?>" 
                            data-status="<?= $status ?>" 
                            data-priority="<?= $priority ?>"
                            data-task-id="<?= $task['task_id'] ?>">
                            <td class="td-checkbox">
                                <input type="checkbox" 
                                       class="task-checkbox" 
                                       <?= ($status === 'completed') ? 'checked' : '' ?>
                                       onchange="toggleTaskStatus(<?= $task['task_id'] ?>, this.checked, 'task')">
                            </td>
                            <td class="td-priority">
                                <span class="priority-badge priority-<?= $priority ?>">
                                    <?= $priorityText ?>
                                </span>
                            </td>
                            <td class="td-task">
                                <div class="task-info">
                                    <div class="task-title"><?= htmlspecialchars($task['task_name']) ?></div>
                                    <?php if (!empty($task['description'])): ?>
                                    <div class="task-description"><?= htmlspecialchars(substr($task['description'], 0, 80)) ?><?= strlen($task['description']) > 80 ? '...' : '' ?></div>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td class="td-project">
                                <span class="project-badge">
                                    <?= htmlspecialchars($project['project_name'] ?? 'Sin proyecto') ?>
                                </span>
                            </td>
                            <td class="td-assigned">
                                <?php if (!empty($task['assigned_to_fullname'])): ?>
                                <span class="assigned-user"><?= htmlspecialchars($task['assigned_to_fullname']) ?></span>
                                <?php else: ?>
                                <span class="no-assigned">Sin asignar</span>
                                <?php endif; ?>
                            </td>
                            <td class="td-due-date">
                                <?php 
                                if ($task['due_date']): 
                                    $dueDate = new DateTime($task['due_date']);
                                    $today = new DateTime('today');
                                    $diff = $today->diff($dueDate);
                                    $diffDays = (int)$diff->format('%R%a');
                                    
                                    if ($diffDays < 0) {
                                        $urgencyClass = 'overdue';
                                        $urgencyText = 'Vencida';
                                    } elseif ($diffDays === 0) {
                                        $urgencyClass = 'today';
                                        $urgencyText = 'Hoy';
                                    } elseif ($diffDays <= 7) {
                                        $urgencyClass = 'week1';
                                        $urgencyText = 'Esta semana';
                                    } else {
                                        $urgencyClass = 'week2';
                                        $urgencyText = $diffDays . ' días';
                                    }
                                ?>
                                <span class="due-date-badge <?= $urgencyClass ?>"><?= $urgencyText ?></span>
                                <?php else: ?>
                                <span class="no-due-date">Sin fecha</span>
                                <?php endif; ?>
                            </td>
                            <td class="td-status">
                                <span class="status-badge status-<?= $status ?>">
                                    <?php 
                                    switch($status) {
                                        case 'in_progress': echo 'En Progreso'; break;
                                        case 'completed': echo 'Completada'; break;
                                        default: echo 'Pendiente'; break;
                                    }
                                    ?>
                                </span>
                            </td>
                            <td class="td-progress">
                                <div class="progress-container-table">
                                    <div class="progress-bar-table">
                                        <div class="progress-fill-table" style="width: <?= intval($task['completion_percentage'] ?? 0) ?>%"></div>
                                    </div>
                                    <span class="progress-text-table"><?= intval($task['completion_percentage'] ?? 0) ?>%</span>
                                </div>
                            </td>
                            <td class="td-actions">
                                <div class="actions-group">
                                    <a href="<?= APP_URL ?>?route=clan_leader/task-details&task_id=<?= $task['task_id'] ?>" class="btn-action-table" title="Ver detalles">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <?php if (!($task['is_subtask'] ?? 0)): ?>
                                    <a href="<?= APP_URL ?>?route=clan_leader/task_edit&task_id=<?= $task['task_id'] ?>" class="btn-action-table" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <a href="#" onclick="deleteTask(<?= $task['task_id'] ?>); return false;" class="btn-action-table" title="Eliminar" style="background: #ef4444 !important;">
                                        <i class="fas fa-trash"></i>
                                    </a>
                                    <a href="#" onclick="cloneTask(<?= $task['task_id'] ?>); return false;" class="btn-action-table" title="Clonar">
                                        <i class="fas fa-copy"></i>
                                    </a>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td class="td-select">
                                <input type="checkbox" 
                                       class="task-checkbox" 
                                       data-task-id="<?= $task['task_id'] ?>" 
                                       data-task-name="<?= htmlspecialchars($task['task_name']) ?>" 
                                       onchange="updateSelection()">
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Paginación -->
        <div class="pagination-container">
            <div class="pagination-info">
                Mostrando <span id="showing-start">1</span> - <span id="showing-end"><?= min(5, count($tasks)) ?></span> de <span id="showing-total"><?= count($tasks) ?></span> tareas
            </div>
            <div class="pagination-buttons" id="paginationButtons">
                <!-- Los botones se generarán con JavaScript -->
            </div>
        </div>
    <?php else: ?>
        <!-- Estado vacío - No hay tareas -->
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-tasks"></i>
            </div>
            <h3>No hay tareas en este proyecto</h3>
            <p>Comienza creando tu primera tarea para este proyecto.</p>
            <button class="btn-nueva-tarea" onclick="openCreateTaskModal(<?= $project['project_id'] ?>)">
                <i class="fas fa-plus"></i>
                Crear Primera Tarea
            </button>
        </div>
    <?php endif; ?>
</div>

<!-- Modal crear tarea - Diseño optimizado -->
<div id="createTaskModal" class="modal-overlay" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <div style="display: flex; align-items: center; gap: 12px;">
                <img src="<?php echo Utils::asset('assets/img/Task_Loader.gif'); ?>" alt="Loading" style="width: 64px; height: 64px;">
                <h3>Nueva Tarea</h3>
            </div>
            <button type="button" class="modal-close" onclick="closeCreateTaskModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form id="createTaskForm" class="modal-body">
            <div class="form-group">
                <label for="task_title">
                    <i class="fas fa-tasks"></i>
                    Título de la Tarea *
                </label>
                <input type="text" id="task_title" name="task_title" class="form-control" required 
                       placeholder="Ej: Revisar documentación del proyecto">
            </div>
            
            <div class="form-group">
                <label for="task_description">
                    <i class="fas fa-align-left"></i>
                    Descripción
                </label>
                <textarea id="task_description" name="task_description" class="form-control" rows="3"
                          placeholder="Descripción detallada de la tarea (opcional)"></textarea>
            </div>

            <div class="grid-row" style="display: grid; grid-template-columns: 1fr 1fr; gap: 16px;">
                <div class="form-group">
                    <label for="task_due_date">
                        <i class="fas fa-calendar"></i>
                        Vencimiento *
                    </label>
                    <input type="date" id="task_due_date" name="task_due_date" class="form-control" required>
                </div>

                <div class="form-group">
                    <label for="task_priority">
                        <i class="fas fa-exclamation-circle"></i>
                        Prioridad
                    </label>
                    <select id="task_priority" name="priority" class="form-control">
                        <option value="low">Baja</option>
                        <option value="medium" selected>Media</option>
                        <option value="high">Alta</option>
                        <option value="critical">Urgente</option>
                    </select>
                </div>
            </div>

            <div class="form-group">
                <label for="task_project">
                    <i class="fas fa-folder"></i>
                    Proyecto
                </label>
                <select id="task_project" name="task_project" class="form-control">
                    <option value="">Seleccionar proyecto...</option>
                </select>
            </div>

            <div class="form-group">
                <label>
                    <i class="fas fa-users"></i>
                    Asignar Colaboradores
                </label>
                
                <div class="member-search">
                    <i class="fas fa-search"></i>
                    <input type="text" class="form-control" placeholder="Buscar colaborador..." onkeyup="filterMembers(this.value)">
                </div>

                <div id="members-container" class="members-grid">
                    <p style="color: #94a3b8; font-size: 0.85rem; margin: 0; grid-column: 1/-1; text-align: center;">Cargando miembros...</p>
                </div>
            </div>
        </form>
        
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closeCreateTaskModal()">
                Cancelar
            </button>
            <button type="button" class="btn-primary" onclick="createTask()">
                Crear Tarea
            </button>
        </div>
    </div>
</div>

<script>
// Variables globales de paginación
let currentPage = 1;
let itemsPerPage = 5;
let allTasksArray = [];

// Función debounce para búsqueda
let searchTimeout;
function debounceSearch() {
    clearTimeout(searchTimeout);
    searchTimeout = setTimeout(() => {
        currentPage = 1; // Resetear a la primera página al buscar
        filterTasks();
    }, 300);
}

// Función para abrir modal de crear tarea
function openCreateTaskModal(projectId = null) {
    console.log('🔄 Abrir modal crear tarea');
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        
        // Establecer fecha mínima como hoy
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('task_due_date').setAttribute('min', today);
        
        // Cargar proyectos y miembros
        loadProjectsAndMembers(projectId);
    }
}

// Función para cerrar modal
function closeCreateTaskModal() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
        // Limpiar formulario
        document.getElementById('createTaskForm').reset();
        // Limpiar selección visual
        document.querySelectorAll('.member-card.selected').forEach(el => el.classList.remove('selected'));
        document.querySelectorAll('input[name="assigned_members[]"]').forEach(el => el.checked = false);
    }
}

// Filtrar miembros
function filterMembers(query) {
    query = query.toLowerCase();
    const cards = document.querySelectorAll('.member-card');
    cards.forEach(card => {
        const name = card.querySelector('.member-name').textContent.toLowerCase();
        if (name.includes(query)) {
            card.style.display = 'flex';
        } else {
            card.style.display = 'none';
        }
    });
}

// Función para cargar proyectos y miembros
function loadProjectsAndMembers(selectedProjectId = null) {
    // Cargar proyectos
    fetch('?route=clan_leader/tasks&action=create&ajax=1')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Llenar select de proyectos
                const projectSelect = document.getElementById('task_project');
                projectSelect.innerHTML = '<option value="">Seleccionar proyecto...</option>';
                if (data.projects && data.projects.length > 0) {
                    data.projects.forEach(project => {
                        const option = document.createElement('option');
                        option.value = project.project_id;
                        option.textContent = project.project_name;
                        if (selectedProjectId && project.project_id == selectedProjectId) {
                            option.selected = true;
                        }
                        projectSelect.appendChild(option);
                    });
                }

                // Llenar contenedor de miembros con nuevo diseño
                const membersContainer = document.getElementById('members-container');
                if (data.members && data.members.length > 0) {
                    membersContainer.innerHTML = '';
                    
                    data.members.forEach(member => {
                        const initials = member.full_name.split(' ').map(n => n[0]).join('').substring(0, 2).toUpperCase();
                        const colors = ['#ef4444', '#f59e0b', '#10b981', '#3b82f6', '#6366f1', '#8b5cf6', '#ec4899'];
                        const color = colors[member.user_id % colors.length];
                        
                        const memberCard = document.createElement('div');
                        memberCard.className = 'member-card';
                        memberCard.onclick = function() { toggleMemberSelection(this, member.user_id); };
                        
                        memberCard.innerHTML = `
                            <div class="member-avatar" style="background: ${color}; color: white;">
                                ${initials}
                            </div>
                            <div class="member-name">${escapeHtml(member.full_name)}</div>
                            <div class="member-check"></div>
                            <input type="checkbox" 
                                   id="member_${member.user_id}" 
                                   name="assigned_members[]" 
                                   value="${member.user_id}"
                                   style="display: none;">
                        `;
                        membersContainer.appendChild(memberCard);
                    });
                } else {
                    membersContainer.innerHTML = '<p style="color: #94a3b8; font-size: 0.85rem; margin: 0; grid-column: 1/-1; text-align: center;">No hay miembros disponibles</p>';
                }
            }
        })
        .catch(error => {
            console.error('Error al cargar proyectos y miembros:', error);
        });
}

// Toggle selección de miembro
function toggleMemberSelection(card, userId) {
    const checkbox = card.querySelector('input[type="checkbox"]');
    checkbox.checked = !checkbox.checked;
    
    if (checkbox.checked) {
        card.classList.add('selected');
    } else {
        card.classList.remove('selected');
    }
}

// Función helper para escapar HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}


// Función para crear tarea
function createTask() {
    const form = document.getElementById('createTaskForm');
    const formData = new FormData(form);
    
    // Validaciones básicas
    const taskTitle = formData.get('task_title');
    const dueDate = formData.get('task_due_date');
    
    if (!taskTitle || !dueDate) {
        alert('Por favor completa los campos requeridos (Título y Fecha de Vencimiento)');
        return;
    }

    // Obtener miembros asignados
    const assignedMembers = [];
    formData.getAll('assigned_members[]').forEach(memberId => {
        if (memberId) {
            assignedMembers.push(parseInt(memberId));
        }
    });

    // Si no hay miembros asignados, asignar al usuario actual
    if (assignedMembers.length === 0) {
        assignedMembers.push(<?= $_SESSION['user_id'] ?? 0 ?>);
    }

    // Preparar datos para enviar
    const taskData = {
        task_title: taskTitle,
        task_description: formData.get('task_description') || '',
        task_due_date: dueDate,
        task_project: formData.get('task_project') || '',
        priority: formData.get('priority') || 'medium',
        assigned_members: assignedMembers,
        subtasks: [],
        labels: []
    };

    // Deshabilitar botones y mostrar carga
    const submitBtn = document.querySelector('#createTaskModal .btn-primary');
    const cancelBtn = document.querySelector('#createTaskModal .btn-secondary');
    const originalBtnText = submitBtn.innerHTML;
    
    submitBtn.disabled = true;
    cancelBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando...';
    submitBtn.style.opacity = '0.7';
    cancelBtn.style.opacity = '0.7';

    // Enviar petición
    fetch('?route=clan_leader/create-task', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(taskData)
    })
    .then(async response => {
        const text = await response.text();
        try {
            return JSON.parse(text);
        } catch(e) {
            console.error('Error parsing response:', text);
            return { success: false, message: 'Respuesta inválida del servidor' };
        }
    })
    .then(data => {
        if (data.success) {
            // Éxito: Mostrar animación y mensaje en el botón
            submitBtn.innerHTML = '<i class="fas fa-check"></i> ¡Creado!';
            submitBtn.style.background = 'linear-gradient(135deg, #059669 0%, #10b981 100%)';
            
            // Mostrar animación de celebración global
            if (typeof showCompletionAnimation === 'function') {
                showCompletionAnimation();
            }
            
            // Esperar un momento antes de cerrar y recargar
            setTimeout(() => {
                closeCreateTaskModal();
                window.location.reload();
            }, 1500);
        } else {
            // Error: Restaurar botones y mostrar alerta
            alert(data.message || 'Error al crear la tarea');
            submitBtn.disabled = false;
            cancelBtn.disabled = false;
            submitBtn.innerHTML = originalBtnText;
            submitBtn.style.opacity = '1';
            cancelBtn.style.opacity = '1';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error de conexión al crear la tarea');
        // Restaurar botones
        submitBtn.disabled = false;
        cancelBtn.disabled = false;
        submitBtn.innerHTML = originalBtnText;
        submitBtn.style.opacity = '1';
        cancelBtn.style.opacity = '1';
    });
}

// Funciones para eliminar y clonar tareas
function deleteTask(taskId) {
    if (!confirm('¿Estás seguro de que deseas eliminar esta tarea?')) {
        return;
    }
    
    fetch('?route=clan_leader/delete-task', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ task_id: taskId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.reload();
        } else {
            alert(data.message || 'Error al eliminar la tarea');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error de conexión');
    });
}

function cloneTask(taskId) {
    fetch('?route=clan_leader/clone-task', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ task_id: taskId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.reload();
        } else {
            alert(data.message || 'Error al clonar la tarea');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error de conexión');
    });
}

// Cerrar modal con ESC
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const modal = document.getElementById('createTaskModal');
        if (modal && modal.style.display === 'flex') {
            closeCreateTaskModal();
        }
    }
});

// Cerrar modal al hacer clic fuera
document.addEventListener('DOMContentLoaded', function() {
    const modal = document.getElementById('createTaskModal');
    if (modal) {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                closeCreateTaskModal();
            }
        });
    }
});

function filterTasks() {
    const statusFilter = document.getElementById('status_filter').value;
    const priorityFilter = document.getElementById('priority_filter').value;
    const searchText = document.getElementById('searchInput').value.toLowerCase();
    const taskRows = document.querySelectorAll('.task-row');
    
    // Filtrar tareas
    let visibleTasks = [];
    taskRows.forEach(row => {
        const status = row.dataset.status;
        const priority = row.dataset.priority;
        const taskName = row.querySelector('.task-title')?.textContent.toLowerCase() || '';
        const taskDesc = row.querySelector('.task-description')?.textContent.toLowerCase() || '';
        
        const statusMatch = !statusFilter || status === statusFilter;
        const priorityMatch = !priorityFilter || priority === priorityFilter;
        const searchMatch = !searchText || taskName.includes(searchText) || taskDesc.includes(searchText);
        
        if (statusMatch && priorityMatch && searchMatch) {
            visibleTasks.push(row);
        }
    });
    
    // Guardar tareas visibles para paginación
    allTasksArray = visibleTasks;
    
    // Aplicar paginación
    applyPagination();
}

function applyPagination() {
    const totalTasks = allTasksArray.length;
    const totalPages = itemsPerPage === 'all' ? 1 : Math.ceil(totalTasks / itemsPerPage);
    
    // Asegurar que currentPage esté en rango válido
    if (currentPage > totalPages && totalPages > 0) {
        currentPage = totalPages;
    }
    if (currentPage < 1) {
        currentPage = 1;
    }
    
    // Ocultar todas las filas primero
    document.querySelectorAll('.task-row').forEach(row => {
        row.style.display = 'none';
    });
    
    // Mostrar solo las filas de la página actual
    if (itemsPerPage === 'all') {
        allTasksArray.forEach(row => {
            row.style.display = 'table-row';
        });
    } else {
        const start = (currentPage - 1) * itemsPerPage;
        const end = start + itemsPerPage;
        
        for (let i = start; i < end && i < allTasksArray.length; i++) {
            allTasksArray[i].style.display = 'table-row';
        }
    }
    
    // Actualizar info de paginación
    updatePaginationInfo(totalTasks, totalPages);
}

function updatePaginationInfo(totalTasks, totalPages) {
    const showingStart = totalTasks === 0 ? 0 : ((currentPage - 1) * itemsPerPage) + 1;
    const showingEnd = itemsPerPage === 'all' ? totalTasks : Math.min(currentPage * itemsPerPage, totalTasks);
    
    document.getElementById('showing-start').textContent = showingStart;
    document.getElementById('showing-end').textContent = showingEnd;
    document.getElementById('showing-total').textContent = totalTasks;
    
    // Generar botones de paginación
    const paginationButtons = document.getElementById('paginationButtons');
    paginationButtons.innerHTML = '';
    
    if (itemsPerPage !== 'all' && totalPages > 1) {
        // Botón anterior
        const prevBtn = document.createElement('button');
        prevBtn.className = 'pagination-btn';
        prevBtn.innerHTML = '<i class="fas fa-chevron-left"></i>';
        prevBtn.disabled = currentPage === 1;
        prevBtn.onclick = () => {
            if (currentPage > 1) {
                currentPage--;
                applyPagination();
            }
        };
        paginationButtons.appendChild(prevBtn);
        
        // Botones de páginas
        for (let i = 1; i <= totalPages; i++) {
            // Mostrar solo algunas páginas alrededor de la actual
            if (i === 1 || i === totalPages || (i >= currentPage - 2 && i <= currentPage + 2)) {
                const pageBtn = document.createElement('button');
                pageBtn.className = 'pagination-btn' + (i === currentPage ? ' active' : '');
                pageBtn.textContent = i;
                pageBtn.onclick = () => {
                    currentPage = i;
                    applyPagination();
                };
                paginationButtons.appendChild(pageBtn);
            } else if (i === currentPage - 3 || i === currentPage + 3) {
                const dots = document.createElement('span');
                dots.textContent = '...';
                dots.style.padding = '0 8px';
                paginationButtons.appendChild(dots);
            }
        }
        
        // Botón siguiente
        const nextBtn = document.createElement('button');
        nextBtn.className = 'pagination-btn';
        nextBtn.innerHTML = '<i class="fas fa-chevron-right"></i>';
        nextBtn.disabled = currentPage === totalPages;
        nextBtn.onclick = () => {
            if (currentPage < totalPages) {
                currentPage++;
                applyPagination();
            }
        };
        paginationButtons.appendChild(nextBtn);
    }
}

function changePerPage() {
    const perPageValue = document.getElementById('perPage').value;
    itemsPerPage = perPageValue === 'all' ? 'all' : parseInt(perPageValue);
    currentPage = 1;
    filterTasks();
}

function resetFilters() {
    document.getElementById('status_filter').value = '';
    document.getElementById('priority_filter').value = '';
    document.getElementById('searchInput').value = '';
    document.getElementById('perPage').value = '5';
    itemsPerPage = 5;
    currentPage = 1;
    filterTasks();
}

// Inicializar paginación al cargar la página
document.addEventListener('DOMContentLoaded', function() {
    filterTasks();
});

function updateStats() {
    // Las estadísticas deben reflejar el total del proyecto, NO los filtros
    // No hacer nada aquí ya que los valores iniciales de PHP son los correctos
    // Esta función se mantiene por compatibilidad pero no modifica los cards
}

function updateTaskStatus(taskId, newStatus, event) {
    // Guardar referencia al select y la fila antes de hacer la petición
    const selectElement = event ? event.target : document.querySelector(`select[onchange*="${taskId}"]`);
    const taskRow = selectElement.closest('.task-row');
    const originalStatus = taskRow.dataset.status;
    
    // Deshabilitar el select mientras se procesa
    selectElement.disabled = true;
    
    const fd = new FormData();
    fd.append('task_id', taskId);
    
    // Determinar qué endpoint usar según el estado
    let endpoint;
    if (newStatus === 'completed' || newStatus === 'pending') {
        fd.append('is_completed', newStatus === 'completed' ? 'true' : 'false');
        endpoint = '?route=clan_leader/toggle-task-status';
    } else {
        fd.append('status', newStatus);
        endpoint = '?route=clan_leader/update-task';
    }
    
    // Hacer la petición
    fetch(endpoint, { 
        method: 'POST', 
        body: fd, 
        credentials: 'same-origin' 
    })
    .then(async response => { 
        const text = await response.text(); 
        try { 
            return JSON.parse(text); 
        } catch(e) { 
            console.error('Error parsing response:', text); 
            return { success: false, message: 'Respuesta inválida del servidor' }; 
        } 
    })
    .then(data => { 
        if (!data.success) { 
            alert(data.message || 'Error al actualizar el estado'); 
            selectElement.value = originalStatus;
        } else {
            // Mostrar animación de completado si cambió a completado
            if (newStatus === 'completed' && originalStatus !== 'completed') {
                showCompletionAnimation();
            }
            
            // Actualizar el DOM localmente
            taskRow.dataset.status = newStatus;
            
            // Actualizar clases visuales
            if (newStatus === 'completed') {
                taskRow.classList.add('completed');
            } else {
                taskRow.classList.remove('completed');
            }
            
            // Mostrar feedback visual temporal
            taskRow.style.backgroundColor = '#d4edda';
            setTimeout(() => {
                taskRow.style.backgroundColor = '';
            }, 1000);
        }
    })
    .catch(error => {
        console.error('Error en la petición:', error);
        alert('Error de conexión al actualizar el estado');
        selectElement.value = originalStatus;
    })
    .finally(() => {
        // Siempre habilitar el select de nuevo
        selectElement.disabled = false;
    });
}

function showCompletionAnimation() {
    console.log('🎉 Mostrando animación de celebración');
    
    // Remover overlay anterior si existe
    const existingOverlay = document.getElementById('completion-animation-overlay');
    if (existingOverlay) {
        existingOverlay.remove();
    }
    
    // Crear overlay
    const overlay = document.createElement('div');
    overlay.id = 'completion-animation-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.4);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 99999;
        opacity: 0;
        transition: opacity 0.3s ease;
    `;
    
    // Crear el elemento lottie
    const lottiePlayer = document.createElement('dotlottie-wc');
    lottiePlayer.setAttribute('src', 'https://lottie.host/3fb875a8-1324-4e18-bb92-116a8e77a2d7/E2KxUaNO1B.lottie');
    lottiePlayer.setAttribute('autoplay', '');
    lottiePlayer.style.cssText = 'width: 350px; height: 350px;';
    
    overlay.appendChild(lottiePlayer);
    document.body.appendChild(overlay);
    
    // Mostrar con animación
    requestAnimationFrame(() => {
        overlay.style.opacity = '1';
    });
    
    // Ocultar después de 3 segundos
    setTimeout(() => {
        overlay.style.opacity = '0';
        setTimeout(() => {
            overlay.remove();
        }, 300);
    }, 3000);
    
    console.log('🎉 Animación creada y mostrada');
}
</script>

<?php
$content = ob_get_clean();
$additionalCSS = [];
require_once __DIR__ . '/../layout.php';
?>
