<?php
// Debug ANTES del buffer para que se ejecute inmediatamente
error_log("Vista projects.php INICIO - Count proyectos: " . (isset($projects) ? count($projects) : 'NO DEFINIDO'));
error_log("Vista projects.php INICIO - ¿Está vacío?: " . (empty($projects) ? 'SÍ' : 'NO'));
error_log("Vista projects.php INICIO - Tipo de variable: " . gettype($projects ?? null));
if (isset($projects) && !empty($projects)) {
    error_log("Vista projects.php INICIO - Primer proyecto ID: " . ($projects[0]['project_id'] ?? 'N/A'));
}

// Guardar el contenido en una variable
ob_start();
?>

<!-- MONDAY.COM THEME - Mismo tema que el dashboard -->
<link rel="stylesheet" href="<?= APP_URL ?>/assets/css/monday-theme.css?v=<?= time() ?>">
<!-- FontAwesome -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">

<div class="monday-layout">
    <main class="monday-main" style="margin-left: 0; width: 100%;">
    <!-- HEADER -->
    <header class="main-header">
        <div class="header-title-row">
            <div>
                <h1 class="page-title">Gestión de Proyectos</h1>
                <p style="color: var(--monday-text-secondary); margin-top: 5px;">Administra proyectos y tareas de <?php echo htmlspecialchars($clan['clan_name'] ?? 'tu clan'); ?></p>
            </div>
            <div class="header-actions">
                <button class="btn-monday btn-primary btn-green" onclick="openCreateProjectModal()">
                    <i class="fas fa-plus"></i>
                    Nuevo Proyecto
                </button>
            </div>
        </div>
        
        <!-- Búsqueda -->
        <div class="search-container-monday" style="margin-top: 20px;">
            <div class="search-input-monday">
                <i class="fas fa-search"></i>
                <input type="text" id="searchInput" value="<?php echo htmlspecialchars($search ?? ''); ?>" 
                       placeholder="Buscar proyectos..." autocomplete="off">
                <button type="button" id="clearSearch" class="btn-clear-monday" style="display: none;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="search-loading-monday" id="searchLoading" style="display: none;">
                <i class="fas fa-spinner fa-spin"></i>
            </div>
        </div>
    </header>

    <!-- Contenido Principal -->
    <div class="board-container">
        <?php 
        // Verificación explícita y debug
        $hasProjects = isset($projects) && is_array($projects) && count($projects) > 0;
        error_log("Vista projects.php - hasProjects: " . ($hasProjects ? 'SÍ' : 'NO'));
        error_log("Vista projects.php - isset projects: " . (isset($projects) ? 'SÍ' : 'NO'));
        error_log("Vista projects.php - is_array: " . (is_array($projects ?? null) ? 'SÍ' : 'NO'));
        error_log("Vista projects.php - count: " . (isset($projects) && is_array($projects) ? count($projects) : 'N/A'));
        ?>
        <?php if ($hasProjects): ?>
            
            <!-- Vista de Lista -->
            <section class="projects-list-view animate-fade-in" id="listView">
                <div class="projects-table-minimal">
                    <div class="table-header-minimal">
                        <div class="header-cell">Proyecto</div>
                        <div class="header-cell">Estado</div>
                        <div class="header-cell">Progreso</div>
                        <div class="header-cell">Tareas</div>
                        <div class="header-cell">Acciones</div>
                    </div>
                    
                    <?php foreach ($projects as $project): ?>
                    <div class="table-row-minimal">
                        <div class="cell-project">
                            <div class="project-icon-list">
                                <i class="fas fa-project-diagram"></i>
                            </div>
                            <div class="project-info-list">
                                <div class="project-name-list"><?= htmlspecialchars($project['project_name']) ?></div>
                                <?php if (!empty($project['description'])): ?>
                                <div class="project-description-list project-tooltip" 
                                     data-tooltip="<?= htmlspecialchars($project['description']) ?>"
                                     style="cursor: help;">
                                    <?= strlen($project['description']) > 30 ? htmlspecialchars(substr($project['description'], 0, 30)) . '...' : htmlspecialchars($project['description']) ?>
                                </div>
                                <?php endif; ?>
                                <?php if (isset($project['kpi_points']) && $project['kpi_points'] > 0): ?>
                                <div class="project-kpi-list">
                                    <i class="fas fa-star"></i>
                                    <?= number_format($project['kpi_points']) ?> KPI
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="cell-status">
                            <span class="status-badge-list status-<?= $project['status'] ?>">
                                <?= ucfirst($project['status']) ?>
                            </span>
                        </div>
                        
                        <div class="cell-progress">
                            <div class="progress-container-list">
                                <div class="progress-bar-list">
                                    <div class="progress-fill-list" style="width: <?= $project['progress_percentage'] ?? 0 ?>%"></div>
                                </div>
                                <span class="progress-text-list"><?= $project['progress_percentage'] ?? 0 ?>%</span>
                            </div>
                        </div>
                        
                        <div class="cell-tasks">
                            <div class="tasks-summary-list">
                                <span class="tasks-total"><?= $project['total_tasks'] ?? 0 ?></span>
                                <span class="tasks-separator">/</span>
                                <span class="tasks-completed"><?= $project['completed_tasks'] ?? 0 ?></span>
                            </div>
                        </div>
                        
                        <div class="cell-actions">
                            <div class="actions-list">
                                <a href="?route=clan_leader/tasks&project_id=<?= $project['project_id'] ?>" class="btn-list-action primary" title="Ver Tareas">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <button class="btn-list-action secondary" onclick="openCreateTaskModal(<?= $project['project_id'] ?>)" title="Tarea">
                                    <i class="fas fa-plus"></i>
                                </button>
                                <div class="dropdown-container">
                                    <button class="btn-list-action menu" onclick="toggleListMenu(<?= $project['project_id'] ?>)" title="Más opciones">
                                        <i class="fas fa-ellipsis-v"></i>
                                    </button>
                                    <div class="dropdown-menu-list" id="listMenu<?= $project['project_id'] ?>">
                                        <button class="menu-item-list" onclick="openEditProjectModal(<?= $project['project_id'] ?>, '<?= htmlspecialchars($project['project_name'], ENT_QUOTES, 'UTF-8') ?>', '<?= htmlspecialchars($project['description'], ENT_QUOTES, 'UTF-8') ?>', '<?= htmlspecialchars($project['time_limit'] ?? '', ENT_QUOTES, 'UTF-8') ?>')">
                                            <i class="fas fa-edit"></i>
                                            Editar
                                        </button>
                                        <button class="menu-item-list" onclick="openAssignProjectModal(<?= $project['project_id'] ?>, '<?= htmlspecialchars($project['project_name'], ENT_QUOTES, 'UTF-8') ?>')">
                                            <i class="fas fa-user-check"></i>
                                            Asignar proyecto
                                        </button>
                                        <button class="menu-item-list" onclick="openCloneProjectModal(<?= $project['project_id'] ?>)">
                                            <i class="fas fa-copy"></i>
                                            Clonar
                                        </button>
                                        <button class="menu-item-list danger" onclick="deleteProject(<?= $project['project_id'] ?>, '<?= htmlspecialchars($project['project_name'], ENT_QUOTES, 'UTF-8') ?>')">
                                            <i class="fas fa-trash"></i>
                                            Eliminar
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Paginación solo para vista lista -->
                <?php if (isset($pagination) && $pagination['viewType'] === 'list' && $pagination['totalPages'] > 1): ?>
                <div class="pagination-container-monday">
                    <div class="pagination-info-monday">
                        Mostrando <?= (($pagination['currentPage'] - 1) * $pagination['itemsPerPage']) + 1 ?> - 
                        <?= min($pagination['currentPage'] * $pagination['itemsPerPage'], $pagination['totalProjects']) ?> 
                        de <?= $pagination['totalProjects'] ?> proyectos
                    </div>
                    <div class="pagination-controls-monday">
                        <button class="btn-pagination-monday" 
                                onclick="goToPage(<?= max(1, $pagination['currentPage'] - 1) ?>)" 
                                <?= $pagination['currentPage'] <= 1 ? 'disabled' : '' ?>>
                            <i class="fas fa-chevron-left"></i>
                            Anterior
                        </button>
                        
                        <div class="pagination-pages-monday">
                            <?php
                            $currentPage = $pagination['currentPage'];
                            $totalPages = $pagination['totalPages'];
                            $startPage = max(1, $currentPage - 2);
                            $endPage = min($totalPages, $currentPage + 2);
                            
                            if ($startPage > 1): ?>
                                <button class="btn-page-monday" onclick="goToPage(1)">1</button>
                                <?php if ($startPage > 2): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                                <button class="btn-page-monday <?= $i === $currentPage ? 'active' : '' ?>" 
                                        onclick="goToPage(<?= $i ?>)">
                                    <?= $i ?>
                                </button>
                            <?php endfor; ?>
                            
                            <?php if ($endPage < $totalPages): ?>
                                <?php if ($endPage < $totalPages - 1): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                                <button class="btn-page-monday" onclick="goToPage(<?= $totalPages ?>)"><?= $totalPages ?></button>
                            <?php endif; ?>
                        </div>
                        
                        <button class="btn-pagination-monday" 
                                onclick="goToPage(<?= min($pagination['totalPages'], $pagination['currentPage'] + 1) ?>)" 
                                <?= $pagination['currentPage'] >= $pagination['totalPages'] ? 'disabled' : '' ?>>
                            Siguiente
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    </div>
                </div>
                
                <!-- GIF decorativo al pie de la tabla (solo si hay proyectos) -->
                <?php if ($hasProjects): ?>
                <div class="table-footer-gif" style="text-align: center; padding: 20px; background: white; border-radius: 0 0 var(--border-radius) var(--border-radius);">
                    <img src="<?php echo Utils::asset('assets/img/proyectos.gif'); ?>" 
                         alt="Proyectos" 
                         style="max-width: 200px; height: auto; opacity: 0.8;">
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </section>
        <?php else: ?>
            <div class="empty-state-monday" style="text-align: center; padding: 60px 20px; background: white; border-radius: var(--border-radius); box-shadow: var(--box-shadow);">
                <div style="font-size: 64px; margin-bottom: 20px;">📋</div>
                <h3 style="font-size: 1.5rem; font-weight: 600; color: var(--monday-text-color); margin-bottom: 8px;">No hay proyectos en el clan</h3>
                <p style="color: var(--monday-text-secondary); margin-bottom: 24px;">Crea tu primer proyecto para comenzar a organizar las tareas del equipo.</p>
                <button class="btn-monday btn-primary btn-green" onclick="openCreateProjectModal()">
                    <i class="fas fa-plus"></i>
                    Crear primer proyecto
                </button>
            </div>
        <?php endif; ?>
    </div>
    </main>
</div>

<!-- Modal para crear proyecto -->
<div id="createProjectModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Crear Nuevo Proyecto</h3>
            <button class="modal-close" onclick="closeCreateProjectModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <form id="createProjectForm" class="modal-form">
                <div class="form-group">
                    <label for="projectName">
                        <i class="fas fa-project-diagram"></i>
                        Nombre del Proyecto
                    </label>
                    <input type="text" id="projectName" name="projectName" required 
                           placeholder="Ingrese el nombre del proyecto">
                </div>
                
                <div class="form-group">
                    <label for="description">
                        <i class="fas fa-align-left"></i>
                        Descripción
                    </label>
                    <textarea id="description" name="description" required 
                              placeholder="Describa el proyecto" rows="4"></textarea>
                </div>
                
                <div class="form-group">
                    <label for="timeLimit">
                        <i class="fas fa-calendar-alt"></i>
                        Fecha Límite
                    </label>
                    <input type="date" id="timeLimit" name="timeLimit" 
                           placeholder="Seleccione la fecha límite del proyecto">
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="isEditable" name="isEditable" value="1">
                        <span class="checkmark"></span>
                        <i class="fas fa-edit"></i>
                        Permitir edición por miembros del clan
                    </label>
                </div>
            </form>
        </div>
        
        <div class="modal-footer">
            <button type="button" class="action-btn secondary" onclick="closeCreateProjectModal()">
                <i class="fas fa-times"></i>
                <span>Cancelar</span>
            </button>
            <button type="submit" form="createProjectForm" class="action-btn primary">
                <i class="fas fa-plus"></i>
                <span>Crear Proyecto</span>
            </button>
        </div>
    </div>
</div>

<!-- Modal para editar proyecto -->
<div id="editProjectModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Editar Proyecto</h3>
            <button class="modal-close" onclick="closeEditProjectModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <form id="editProjectForm" class="modal-form">
                <input type="hidden" id="editProjectId" name="projectId">
                
                <div class="form-group">
                    <label for="editProjectName">
                        <i class="fas fa-project-diagram"></i>
                        Nombre del Proyecto
                    </label>
                    <input type="text" id="editProjectName" name="projectName" required 
                           placeholder="Ingrese el nombre del proyecto">
                </div>
                
                <div class="form-group">
                    <label for="editDescription">
                        <i class="fas fa-align-left"></i>
                        Descripción
                    </label>
                    <textarea id="editDescription" name="description" required 
                              placeholder="Describa el proyecto" rows="4"></textarea>
                </div>
                
                <div class="form-group">
                    <label for="editTimeLimit">
                        <i class="fas fa-calendar-alt"></i>
                        Fecha Límite
                    </label>
                    <input type="date" id="editTimeLimit" name="timeLimit" 
                           placeholder="Seleccione la fecha límite del proyecto">
                </div>
            </form>
        </div>
        
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closeEditProjectModal()">
                <i class="fas fa-times"></i>
                Cancelar
            </button>
            <button type="submit" form="editProjectForm" class="btn-primary">
                <i class="fas fa-save"></i>
                Guardar Cambios
            </button>
        </div>
    </div>
</div>

<!-- Modal para asignar proyecto -->
<div id="assignProjectModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3><i class="fas fa-user-check"></i> Asignar Proyecto</h3>
            <button class="modal-close" onclick="closeAssignProjectModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <div class="modal-body">
            <form id="assignProjectForm" class="modal-form">
                <input type="hidden" id="assignProjectId" name="projectId">
                
                <div class="form-group">
                    <label>
                        <i class="fas fa-info-circle"></i>
                        Proyecto: <strong id="assignProjectName"></strong>
                    </label>
                    <p style="font-size: 13px; color: #6b7280; margin-top: 5px;">
                        Selecciona los usuarios que tendrán acceso completo a este proyecto
                    </p>
                </div>
                
                <div class="form-group">
                    <label>
                        <i class="fas fa-search"></i>
                        Buscar usuarios
                    </label>
                    <input type="text" 
                           id="userSearchInput" 
                           class="form-control" 
                           placeholder="Buscar por nombre o email..." 
                           style="padding: 10px 12px; border: 1px solid #e5e7eb; border-radius: 8px; width: 100%; font-size: 14px;">
                </div>
                
                <div class="form-group">
                    <label>
                        <i class="fas fa-users"></i>
                        Usuarios disponibles (<span id="userCount">0</span>)
                    </label>
                    <div id="usersListContainer" style="max-height: 350px; overflow-y: auto; border: 1px solid #e5e7eb; border-radius: 8px; padding: 12px;">
                        <div style="text-align: center; padding: 20px; color: #6b7280;">
                            <i class="fas fa-spinner fa-spin"></i> Cargando usuarios...
                        </div>
                    </div>
                </div>
            </form>
        </div>
        
        <div class="modal-footer">
            <button type="button" class="btn-secondary" onclick="closeAssignProjectModal()">
                <i class="fas fa-times"></i>
                Cancelar
            </button>
            <button type="button" class="btn-primary" onclick="submitAssignProject()">
                <i class="fas fa-check"></i>
                Asignar Usuarios
            </button>
        </div>
    </div>
</div>

<script>
// Funciones para el modal de crear proyecto
function openCreateProjectModal() {
    document.getElementById('createProjectModal').style.display = 'flex';
}

function closeCreateProjectModal() {
    document.getElementById('createProjectModal').style.display = 'none';
    document.getElementById('createProjectForm').reset();
}

// Funciones para el modal de editar proyecto
function openEditProjectModal(projectId, projectName, description, timeLimit) {
    console.log('openEditProjectModal llamado con:', projectId, projectName, description, timeLimit);
    document.getElementById('editProjectId').value = projectId;
    document.getElementById('editProjectName').value = projectName;
    document.getElementById('editDescription').value = description;
    document.getElementById('editTimeLimit').value = timeLimit || '';
    document.getElementById('editProjectModal').style.display = 'flex';
}

function closeEditProjectModal() {
    document.getElementById('editProjectModal').style.display = 'none';
    document.getElementById('editProjectForm').reset();
}

// Funciones para el modal de asignar proyecto
function openAssignProjectModal(projectId, projectName) {
    console.log('openAssignProjectModal llamado con:', projectId, projectName);
    document.getElementById('assignProjectId').value = projectId;
    document.getElementById('assignProjectName').textContent = projectName;
    document.getElementById('assignProjectModal').style.display = 'flex';
    
    // Cargar usuarios del clan
    loadClanUsersForProject(projectId);
}

function closeAssignProjectModal() {
    document.getElementById('assignProjectModal').style.display = 'none';
    document.getElementById('assignProjectForm').reset();
}

// Cargar usuarios del clan para asignar al proyecto
function loadClanUsersForProject(projectId) {
    const container = document.getElementById('usersListContainer');
    container.innerHTML = '<div style="text-align: center; padding: 20px; color: #6b7280;"><i class="fas fa-spinner fa-spin"></i> Cargando usuarios...</div>';
    
    fetch('?route=clan_leader/get-clan-users-for-project&project_id=' + projectId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayUsersForAssignment(data.users);
            } else {
                container.innerHTML = '<div style="text-align: center; padding: 20px; color: #e74c3c;"><i class="fas fa-exclamation-circle"></i> ' + data.message + '</div>';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            container.innerHTML = '<div style="text-align: center; padding: 20px; color: #e74c3c;"><i class="fas fa-exclamation-circle"></i> Error al cargar usuarios</div>';
        });
}

// Variable global para almacenar todos los usuarios
let allUsersData = [];

// Mostrar lista de usuarios con checkboxes
function displayUsersForAssignment(users) {
    const container = document.getElementById('usersListContainer');
    allUsersData = users; // Guardar para búsqueda
    
    if (users.length === 0) {
        container.innerHTML = '<div style="text-align: center; padding: 20px; color: #6b7280;"><i class="fas fa-info-circle"></i> No hay usuarios disponibles en el sistema</div>';
        document.getElementById('userCount').textContent = '0';
        return;
    }
    
    renderUsersList(users);
    updateUserCount(users.length);
    
    // Configurar buscador
    const searchInput = document.getElementById('userSearchInput');
    searchInput.value = ''; // Limpiar búsqueda anterior
    searchInput.addEventListener('input', function() {
        filterUsers(this.value);
    });
}

// Renderizar lista de usuarios
function renderUsersList(users) {
    const container = document.getElementById('usersListContainer');
    
    if (users.length === 0) {
        container.innerHTML = '<div style="text-align: center; padding: 20px; color: #6b7280;"><i class="fas fa-search"></i> No se encontraron usuarios con ese criterio</div>';
        return;
    }
    
    let html = '<div style="display: flex; flex-direction: column; gap: 12px;">';
    
    users.forEach(user => {
        const isAssigned = user.is_assigned == 1;
        const isInClan = user.is_in_clan == 1;
        const clanBadge = isInClan ? 
            '<span style="background: rgba(102, 126, 234, 0.1); color: #667eea; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: 600;"><i class="fas fa-users"></i> ' + escapeHtml(user.clan_name) + '</span>' : 
            '<span style="background: rgba(127, 140, 141, 0.1); color: #7f8c8d; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: 600;">' + escapeHtml(user.clan_name) + '</span>';
        
        html += `
            <label class="user-checkbox-item" 
                   data-user-name="${escapeHtml(user.full_name).toLowerCase()}" 
                   data-user-email="${escapeHtml(user.email).toLowerCase()}"
                   data-user-clan="${escapeHtml(user.clan_name).toLowerCase()}"
                   style="display: flex; align-items: center; gap: 12px; padding: 12px; border: 2px solid ${isAssigned ? '#667eea' : '#e5e7eb'}; border-radius: 8px; cursor: pointer; transition: all 0.2s ease; background: ${isAssigned ? 'rgba(102, 126, 234, 0.05)' : 'white'};">
                <input type="checkbox" 
                       name="user_ids[]" 
                       value="${user.user_id}" 
                       ${isAssigned ? 'checked' : ''}
                       style="width: 20px; height: 20px; cursor: pointer;">
                <div style="flex: 1;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                        <span style="font-weight: 600; color: #2c3e50;">
                            ${escapeHtml(user.full_name)}
                        </span>
                        ${clanBadge}
                    </div>
                    <div style="font-size: 12px; color: #7f8c8d;">
                        ${escapeHtml(user.email)}
                    </div>
                </div>
                ${isAssigned ? '<span style="color: #667eea; font-size: 12px; font-weight: 600;"><i class="fas fa-check-circle"></i> Asignado</span>' : ''}
            </label>
        `;
    });
    
    html += '</div>';
    container.innerHTML = html;
    
    // Agregar evento para cambiar el estilo al hacer click
    container.querySelectorAll('.user-checkbox-item').forEach(label => {
        const checkbox = label.querySelector('input[type="checkbox"]');
        checkbox.addEventListener('change', function() {
            if (this.checked) {
                label.style.border = '2px solid #667eea';
                label.style.background = 'rgba(102, 126, 234, 0.05)';
            } else {
                label.style.border = '2px solid #e5e7eb';
                label.style.background = 'white';
            }
        });
    });
}

// Filtrar usuarios por búsqueda
function filterUsers(searchTerm) {
    if (!searchTerm.trim()) {
        renderUsersList(allUsersData);
        updateUserCount(allUsersData.length);
        return;
    }
    
    const search = searchTerm.toLowerCase();
    const filteredUsers = allUsersData.filter(user => {
        return user.full_name.toLowerCase().includes(search) ||
               user.email.toLowerCase().includes(search) ||
               user.clan_name.toLowerCase().includes(search);
    });
    
    renderUsersList(filteredUsers);
    updateUserCount(filteredUsers.length);
}

// Actualizar contador de usuarios
function updateUserCount(count) {
    document.getElementById('userCount').textContent = count;
}

// Enviar asignación de usuarios al proyecto
function submitAssignProject() {
    const projectId = document.getElementById('assignProjectId').value;
    const checkboxes = document.querySelectorAll('#usersListContainer input[type="checkbox"]:checked');
    const userIds = Array.from(checkboxes).map(cb => cb.value);
    
    if (userIds.length === 0) {
        showToast('Por favor selecciona al menos un usuario', 'warning');
        return;
    }
    
    const formData = new FormData();
    formData.append('project_id', projectId);
    formData.append('user_ids', JSON.stringify(userIds));
    
    // Mostrar loading en el botón
    const submitBtn = document.querySelector('#assignProjectModal .btn-primary');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Asignando...';
    submitBtn.disabled = true;
    
    fetch('?route=clan_leader/assign-users-to-project', {
        method: 'POST',
        credentials: 'same-origin',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            closeAssignProjectModal();
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showToast(data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error de conexión', 'error');
    })
    .finally(() => {
        // Restaurar botón
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Eliminar proyecto
function deleteProject(projectId, projectName) {
    console.log('deleteProject llamado con:', projectId, projectName);
    
    showConfirmationModal({
        title: 'Eliminar Proyecto',
        message: `¿Estás seguro de que quieres eliminar el proyecto "${projectName}"?\n\nEsta acción no se puede deshacer.`,
        type: 'danger',
        confirmText: 'Eliminar',
        cancelText: 'Cancelar',
        onConfirm: () => {
            const formData = new FormData();
            formData.append('projectId', projectId);
            
            fetch('?route=clan_leader/delete-project', {
                method: 'POST',
                credentials: 'same-origin',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showToast(data.message, 'success');
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    showToast(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('Error de conexión', 'error');
            });
        }
    });
}

// Función para mostrar modal de confirmación personalizado
function showConfirmationModal(options) {
    const {
        title = 'Confirmar Acción',
        message = '¿Estás seguro de que quieres realizar esta acción?',
        type = 'warning',
        confirmText = 'Confirmar',
        cancelText = 'Cancelar',
        onConfirm = null,
        onCancel = null
    } = options;

    // Crear el HTML del modal
    const modalHTML = `
        <div class="confirmation-modal-overlay" id="confirmationModalOverlay">
            <div class="confirmation-modal" id="confirmationModal">
                <div class="confirmation-modal-header">
                    <h3 class="confirmation-modal-title">
                        <i class="fas fa-${getIconForType(type)}"></i>
                        ${title}
                    </h3>
                </div>
                <div class="confirmation-modal-body">
                    <i class="fas fa-${getIconForType(type)} confirmation-modal-icon ${type}"></i>
                    <p class="confirmation-modal-message">${message}</p>
                    <div class="confirmation-modal-actions">
                        <button class="confirmation-modal-btn cancel" id="confirmationCancelBtn">
                            <i class="fas fa-times"></i>
                            ${cancelText}
                        </button>
                        <button class="confirmation-modal-btn confirm" id="confirmationConfirmBtn">
                            <i class="fas fa-trash"></i>
                            ${confirmText}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;

    // Agregar el modal al DOM
    document.body.insertAdjacentHTML('beforeend', modalHTML);

    const overlay = document.getElementById('confirmationModalOverlay');
    const modal = document.getElementById('confirmationModal');
    const confirmBtn = document.getElementById('confirmationConfirmBtn');
    const cancelBtn = document.getElementById('confirmationCancelBtn');

    // Mostrar el modal con animación
    setTimeout(() => {
        overlay.classList.add('show');
    }, 10);

    // Función para cerrar el modal
    const closeModal = (result) => {
        overlay.classList.remove('show');
        setTimeout(() => {
            document.body.removeChild(overlay);
            if (result && onConfirm) {
                onConfirm();
            } else if (!result && onCancel) {
                onCancel();
            }
        }, 300);
    };

    // Event listeners
    confirmBtn.addEventListener('click', () => closeModal(true));
    cancelBtn.addEventListener('click', () => closeModal(false));
    
    // Cerrar al hacer clic en el overlay
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
            closeModal(false);
        }
    });

    // Cerrar con Escape
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeModal(false);
            document.removeEventListener('keydown', handleEscape);
        }
    };
    
    document.addEventListener('keydown', handleEscape);
}

// Función auxiliar para obtener el icono según el tipo
function getIconForType(type) {
    switch (type) {
        case 'danger':
            return 'exclamation-triangle';
        case 'warning':
            return 'exclamation-triangle';
        case 'info':
            return 'info-circle';
        case 'success':
            return 'check-circle';
        default:
            return 'question-circle';
    }
}

// Manejar envío del formulario de crear proyecto
document.getElementById('createProjectForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    fetch('?route=clan_leader/create-project', {
        method: 'POST',
        credentials: 'same-origin',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            closeCreateProjectModal();
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showToast(data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error de conexión', 'error');
    });
});

// Manejar envío del formulario de editar proyecto
document.getElementById('editProjectForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    fetch('?route=clan_leader/update-project', {
        method: 'POST',
        credentials: 'same-origin',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            closeEditProjectModal();
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showToast(data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error de conexión', 'error');
    });
});

// Cerrar modales al hacer clic fuera
document.getElementById('createProjectModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeCreateProjectModal();
    }
});

document.getElementById('editProjectModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeEditProjectModal();
    }
});

document.getElementById('assignProjectModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeAssignProjectModal();
    }
});


// ========== NUEVAS FUNCIONALIDADES MEJORADAS ==========

// Función para intercambiar entre vista de cards y lista

// Función para ir a una página específica
function goToPage(page) {
    const url = new URL(window.location);
    url.searchParams.set('page', page);
    const search = document.getElementById('searchInput').value;
    if (search) {
        url.searchParams.set('search', search);
    }
    url.searchParams.set('view', 'list');
    window.location.href = url.toString();
}


// Abrir modal para crear tarea
function openCreateTaskModal(projectId) {
    // Redirigir a la página de creación de tareas con el proyecto preseleccionado
    window.location.href = `?route=clan_leader/tasks&action=create&project_id=${projectId}`;
}

// Función para mostrar mensajes toast
function showToast(message, type = 'info') {
    // Crear elemento toast si no existe
    let toast = document.getElementById('toast');
    if (!toast) {
        toast = document.createElement('div');
        toast.id = 'toast';
        toast.className = 'toast';
        document.body.appendChild(toast);
    }
    
    // Configurar mensaje y tipo
    toast.textContent = message;
    toast.className = `toast toast-${type} show`;
    
    // Ocultar después de 3 segundos
    setTimeout(() => {
        toast.classList.remove('show');
    }, 5000);
}

// Función para clonar proyecto
function openCloneProjectModal(projectId) {
    // Cargar datos del proyecto para mostrar en el modal
    fetch('?route=clan_leader/get-project-data&project_id=' + projectId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showCloneProjectModal(data.project);
            } else {
                showToast('Error al cargar los datos del proyecto: ' + data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('Error de conexión al cargar los datos del proyecto', 'error');
        });
}

// Función para mostrar el modal de clonación de proyectos
function showCloneProjectModal(project) {
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-content clone-project-modal">
            <div class="modal-header">
                <h3><i class="fas fa-copy"></i> Clonar Proyecto</h3>
                <button class="modal-close" onclick="closeCloneProjectModal()">&times;</button>
            </div>
            <div class="modal-body">
                <form id="cloneProjectForm">
                    <input type="hidden" id="originalProjectId" value="${project.project_id}">
                    
                    <div class="form-group">
                        <label for="cloneProjectName">Nombre del proyecto</label>
                        <input type="text" id="cloneProjectName" name="project_name" value="${project.project_name} (Copia)" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="cloneProjectDescription">Descripción</label>
                        <textarea id="cloneProjectDescription" name="description" rows="3">${project.description || ''}</textarea>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="cloneProjectStartDate">Fecha de inicio</label>
                            <input type="date" id="cloneProjectStartDate" name="start_date" value="${new Date().toISOString().split('T')[0]}">
                        </div>
                        
                        <div class="form-group">
                            <label for="cloneProjectEndDate">Fecha de fin</label>
                            <input type="date" id="cloneProjectEndDate" name="end_date" value="${project.time_limit || ''}">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="cloneTasks" name="clone_tasks" checked>
                            Clonar también las tareas y subtareas del proyecto
                        </label>
                    </div>
                    
                    <div class="form-group" id="adjustDatesGroup" style="display: none;">
                        <label>
                            <input type="checkbox" id="adjustDates" name="adjust_dates" checked>
                            Ajustar fechas de tareas proporcionalmente según la nueva duración del proyecto
                        </label>
                        <small class="form-help">Las fechas de las tareas se ajustarán automáticamente para mantener la proporción original del proyecto.</small>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-secondary" onclick="closeCloneProjectModal()">Cancelar</button>
                <button type="button" class="btn-primary" onclick="cloneProject()">
                    <i class="fas fa-copy"></i> Clonar Proyecto
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    modal.style.display = 'flex';
    
    // Mostrar/ocultar opción de ajuste de fechas según si se clonan tareas
    const cloneTasksCheckbox = document.getElementById('cloneTasks');
    const adjustDatesGroup = document.getElementById('adjustDatesGroup');
    
    cloneTasksCheckbox.addEventListener('change', function() {
        adjustDatesGroup.style.display = this.checked ? 'block' : 'none';
    });
}

// Función para cerrar el modal de clonación de proyectos
function closeCloneProjectModal() {
    const modal = document.querySelector('.modal-overlay');
    if (modal) {
        modal.remove();
    }
}

// Función para ejecutar la clonación de proyectos
function cloneProject() {
    const form = document.getElementById('cloneProjectForm');
    const formData = new FormData(form);
    
    // Agregar campos adicionales
    formData.append('originalProjectId', document.getElementById('originalProjectId').value);
    formData.append('clone_tasks', document.getElementById('cloneTasks').checked ? '1' : '0');
    formData.append('adjust_dates', document.getElementById('adjustDates').checked ? '1' : '0');
    
    // Mostrar loading
    const submitBtn = document.querySelector('.btn-primary');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Clonando...';
    submitBtn.disabled = true;
    
    fetch('?route=clan_leader/clone-project', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            closeCloneProjectModal();
            showToast('Proyecto clonado exitosamente', 'success');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showToast('Error al clonar el proyecto: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error de conexión al clonar el proyecto', 'error');
    })
    .finally(() => {
        // Restaurar botón
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Función para toggle del menú de lista (posicionamiento absoluto al viewport)
function toggleListMenu(projectId) {
    const menu = document.getElementById('listMenu' + projectId);
    const button = document.querySelector(`[onclick*="toggleListMenu(${projectId})"]`);
    const allMenus = document.querySelectorAll('.dropdown-menu-list');
    
    // Cerrar todos los otros menús
    allMenus.forEach(m => {
        if (m !== menu) {
            m.style.display = 'none';
            m.style.position = 'absolute';
        }
    });
    
    // Toggle del menú actual
    if (menu.style.display === 'block') {
        menu.style.display = 'none';
        menu.style.position = 'absolute';
    } else {
        // Obtener posición del botón relativa al viewport
        const buttonRect = button.getBoundingClientRect();
        
        // Posicionar el menú de forma fija respecto al viewport
        menu.style.position = 'fixed';
        menu.style.top = (buttonRect.top - 200) + 'px'; // 200px arriba del botón
        menu.style.left = (buttonRect.right - 180) + 'px'; // Alineado a la derecha del botón
        menu.style.zIndex = '999999';
        menu.style.display = 'block';
    }
}

// Cerrar menús al hacer clic en cualquier opción
function handleMenuItemClick(callback) {
    return function(...args) {
        // Cerrar todos los menús
        closeAllHorizontalMenus();
        // Ejecutar la función original
        if (typeof callback === 'function') {
            callback.apply(this, args);
        }
    };
}

// Función para el menú de la vista cards (mantener compatibilidad)
function toggleProjectMenu(projectId) {
    console.log('toggleProjectMenu llamado con:', projectId);
    const menu = document.getElementById(`projectMenu${projectId}`);
    
    if (!menu) {
        console.error(`No se encontró el menú con ID: projectMenu${projectId}`);
        return;
    }
    
    // Cerrar todos los otros menús de cards
    const allCardMenus = document.querySelectorAll('.dropdown-menu-minimal');
    allCardMenus.forEach(m => {
        if (m !== menu) {
            m.style.display = 'none';
        }
    });
    
    // Toggle del menú actual
    if (menu.style.display === 'block') {
        menu.style.display = 'none';
    } else {
        menu.style.display = 'block';
    }
}

// Cerrar menús al hacer clic fuera
document.addEventListener('click', function(e) {
    if (!e.target.closest('.dropdown-container')) {
        const allMenus = document.querySelectorAll('.dropdown-menu-list');
        allMenus.forEach(menu => {
            menu.style.display = 'none';
            menu.style.position = 'fixed';
        });
    }
});

// Toggle delegación de proyecto
function toggleProjectDelegation(projectId, isAllowed) {
    const formData = new FormData();
    formData.append('project_id', projectId);
    formData.append('allow_delegation', isAllowed ? '1' : '0');
    
    fetch('?route=clan_leader/update-project-delegation', {
        method: 'POST',
        credentials: 'same-origin',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message || 'Configuración actualizada', 'success');
            
            // Actualizar visualmente el checkbox
            const checkmark = document.querySelector(`[data-project-id="${projectId}"] + .checkmark-custom`);
            if (checkmark) {
                if (isAllowed) {
                    checkmark.style.background = 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
                    checkmark.style.borderColor = '#667eea';
                } else {
                    checkmark.style.background = '#fff';
                    checkmark.style.borderColor = '#d1d5db';
                }
            }
        } else {
            showToast(data.message || 'Error al actualizar configuración', 'error');
            // Revertir checkbox si hay error
            const checkbox = document.querySelector(`[data-project-id="${projectId}"]`);
            if (checkbox) {
                checkbox.checked = !isAllowed;
            }
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Error de conexión', 'error');
        // Revertir checkbox si hay error
        const checkbox = document.querySelector(`[data-project-id="${projectId}"]`);
        if (checkbox) {
            checkbox.checked = !isAllowed;
        }
    });
}

// ========== BÚSQUEDA EN TIEMPO REAL ==========

let searchTimeout;
let isSearching = false;

// Función para realizar búsqueda con debounce
function performSearch(searchTerm) {
    console.log('performSearch llamado con:', searchTerm);
    
    // Limpiar timeout anterior
    if (searchTimeout) {
        clearTimeout(searchTimeout);
    }
    
    // Si el término está vacío, mostrar todos los proyectos
    if (!searchTerm.trim()) {
        console.log('Término vacío, limpiando búsqueda');
        clearSearch();
        return;
    }
    
    console.log('Iniciando búsqueda para:', searchTerm);
    
    // Mostrar loading
    showSearchLoading(true);
    
    // Configurar timeout para evitar demasiadas peticiones
    searchTimeout = setTimeout(() => {
        console.log('Enviando petición AJAX...');
        
        fetch('?route=clan_leader/search-projects-ajax', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            credentials: 'same-origin',
            body: `search=${encodeURIComponent(searchTerm)}`
        })
        .then(response => {
            console.log('Respuesta recibida:', response.status, response.statusText);
            return response.json();
        })
        .then(data => {
            console.log('Datos recibidos:', data);
            if (data.success) {
                // Si estamos en vista lista, redirigir con parámetros de búsqueda y paginación
                const url = new URL(window.location);
                if (url.searchParams.get('view') === 'list') {
                    url.searchParams.set('search', searchTerm);
                    url.searchParams.set('view', 'list');
                    url.searchParams.set('page', '1');
                    window.location.href = url.toString();
                } else {
                    updateProjectsDisplay(data.projects);
                    updateURL(searchTerm);
                }
            } else {
                console.error('Error en respuesta:', data.message);
                showToast('Error en la búsqueda: ' + data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error en búsqueda:', error);
            showToast('Error de conexión en la búsqueda', 'error');
        })
        .finally(() => {
            showSearchLoading(false);
        });
    }, 300); // 300ms de debounce
}

// Función para limpiar búsqueda
function clearSearch() {
    const searchInput = document.getElementById('searchInput');
    const clearBtn = document.getElementById('clearSearch');
    
    searchInput.value = '';
    clearBtn.style.display = 'none';
    
    // Recargar la página sin parámetros de búsqueda, pero manteniendo view si está en lista
    const url = new URL(window.location);
    url.searchParams.delete('search');
    url.searchParams.delete('page');
    const viewType = url.searchParams.get('view');
    if (viewType !== 'list') {
        url.searchParams.delete('view');
    }
    window.location.href = url.toString();
}

// Función para mostrar/ocultar loading
function showSearchLoading(show) {
    const loading = document.getElementById('searchLoading');
    const searchInput = document.getElementById('searchInput');
    
    if (show) {
        loading.style.display = 'block';
        searchInput.style.paddingRight = '60px';
        isSearching = true;
    } else {
        loading.style.display = 'none';
        searchInput.style.paddingRight = '40px';
        isSearching = false;
    }
}

// Función para actualizar la URL sin recargar la página
function updateURL(searchTerm) {
    const url = new URL(window.location);
    if (searchTerm.trim()) {
        url.searchParams.set('search', searchTerm);
        // Si está en vista lista, mantener view y resetear página
        if (url.searchParams.get('view') === 'list') {
            url.searchParams.set('page', '1');
        }
    } else {
        url.searchParams.delete('search');
    }
    window.history.pushState({}, '', url);
}

// Función para actualizar la visualización de proyectos
function updateProjectsDisplay(projects) {
    const contentMinimal = document.querySelector('.content-minimal');
    
    if (projects.length === 0) {
        // Mostrar estado vacío
        contentMinimal.innerHTML = `
            <div class="empty-minimal">
                <div class="empty-icon-minimal">
                    🔍
                </div>
                <h3>No se encontraron proyectos</h3>
                <p>No hay proyectos que coincidan con tu búsqueda.</p>
                <button class="btn-minimal secondary" onclick="clearSearch()">
                    <i class="fas fa-times"></i>
                    Limpiar búsqueda
                </button>
            </div>
        `;
        return;
    }
    
    // Solo actualizar vista lista
    updateListView(projects);
}

// Función para actualizar vista de cards

// Función para actualizar vista de lista
function updateListView(projects) {
    const listView = document.getElementById('listView');
    const projectsTable = listView.querySelector('.projects-table-minimal');
    const headerRow = projectsTable.querySelector('.table-header-minimal');
    
    const rowsHtml = projects.map(project => `
        <div class="table-row-minimal">
            <div class="cell-project">
                <div class="project-icon-list">
                    <i class="fas fa-project-diagram"></i>
                </div>
                <div class="project-info-list">
                    <div class="project-name-list">${escapeHtml(project.project_name)}</div>
                    ${project.description ? `<div class="project-description-list project-tooltip" data-tooltip="${escapeHtml(project.description)}" style="cursor: help;">${project.description.length > 30 ? escapeHtml(project.description.substring(0, 30)) + '...' : escapeHtml(project.description)}</div>` : ''}
                    ${project.kpi_points && project.kpi_points > 0 ? `
                    <div class="project-kpi-list">
                        <i class="fas fa-star"></i>
                        ${formatNumber(project.kpi_points)} KPI
                    </div>
                    ` : ''}
                </div>
            </div>
            
            <div class="cell-status">
                <span class="status-badge-list status-${project.status}">
                    ${capitalizeFirst(project.status)}
                </span>
            </div>
            
            <div class="cell-progress">
                <div class="progress-container-list">
                    <div class="progress-bar-list">
                        <div class="progress-fill-list" style="width: ${project.progress_percentage || 0}%"></div>
                    </div>
                    <span class="progress-text-list">${project.progress_percentage || 0}%</span>
                </div>
            </div>
            
            <div class="cell-tasks">
                <div class="tasks-summary-list">
                    <span class="tasks-total">${project.total_tasks || 0}</span>
                    <span class="tasks-separator">/</span>
                    <span class="tasks-completed">${project.completed_tasks || 0}</span>
                </div>
            </div>
            
            <div class="cell-actions">
                <div class="actions-list">
                    <a href="?route=clan_leader/tasks&project_id=${project.project_id}" class="btn-list-action primary" title="Ver Tareas">
                        <i class="fas fa-eye"></i>
                    </a>
                    <button class="btn-list-action secondary" onclick="openCreateTaskModal(${project.project_id})" title="Tarea">
                        <i class="fas fa-plus"></i>
                    </button>
                    <div class="dropdown-container">
                        <button class="btn-list-action menu" onclick="toggleListMenu(${project.project_id})" title="Más opciones">
                            <i class="fas fa-ellipsis-v"></i>
                        </button>
                        <div class="dropdown-menu-list" id="listMenu${project.project_id}">
                            <button class="menu-item-list" onclick="openEditProjectModal(${project.project_id}, ${JSON.stringify(project.project_name)}, ${JSON.stringify(project.description)}, ${JSON.stringify(project.time_limit || '')})">
                                <i class="fas fa-edit"></i>
                                Editar
                            </button>
                            <button class="menu-item-list" onclick="openAssignProjectModal(${project.project_id}, ${JSON.stringify(project.project_name)})">
                                <i class="fas fa-user-check"></i>
                                Asignar proyecto
                            </button>
                            <button class="menu-item-list" onclick="openCloneProjectModal(${project.project_id})">
                                <i class="fas fa-copy"></i>
                                Clonar
                            </button>
                            <button class="menu-item-list danger" onclick="deleteProject(${project.project_id}, ${JSON.stringify(project.project_name)})">
                                <i class="fas fa-trash"></i>
                                Eliminar
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `).join('');
    
    // Reemplazar todas las filas excepto el header
    const allRows = projectsTable.querySelectorAll('.table-row-minimal');
    allRows.forEach(row => row.remove());
    
    // Insertar las nuevas filas después del header
    headerRow.insertAdjacentHTML('afterend', rowsHtml);
    
    // Reinicializar tooltips después de actualizar el contenido
    initCustomTooltips();
    
    // Agregar event listeners para cerrar menús al hacer clic en opciones
    document.querySelectorAll('.dropdown-menu-list .menu-item-list').forEach(item => {
        item.addEventListener('click', function() {
            setTimeout(() => {
                const allMenus = document.querySelectorAll('.dropdown-menu-list');
                allMenus.forEach(menu => {
                    menu.style.display = 'none';
                    menu.style.position = 'fixed';
                });
            }, 100);
        });
    });
}

// Funciones auxiliares
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function capitalizeFirst(str) {
    return str.charAt(0).toUpperCase() + str.slice(1);
}

function formatNumber(num) {
    return new Intl.NumberFormat().format(num);
}

// Inicializar eventos de búsqueda
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('searchInput');
    const clearBtn = document.getElementById('clearSearch');
    
    // Evento para búsqueda en tiempo real
    searchInput.addEventListener('input', function(e) {
        const searchTerm = e.target.value;
        console.log('Input event disparado:', searchTerm);
        
        // Mostrar/ocultar botón limpiar
        if (searchTerm.trim()) {
            clearBtn.style.display = 'block';
        } else {
            clearBtn.style.display = 'none';
        }
        
        // Realizar búsqueda
        performSearch(searchTerm);
    });
    
    // Evento para botón limpiar
    clearBtn.addEventListener('click', function() {
        clearSearch();
    });
    
    // Mostrar botón limpiar si hay búsqueda inicial
    if (searchInput.value.trim()) {
        clearBtn.style.display = 'block';
    }
    
    // Manejar navegación del navegador (atrás/adelante)
    window.addEventListener('popstate', function(e) {
        const urlParams = new URLSearchParams(window.location.search);
        const searchParam = urlParams.get('search') || '';
        searchInput.value = searchParam;
        
        if (searchParam.trim()) {
            clearBtn.style.display = 'block';
            performSearch(searchParam);
        } else {
            clearBtn.style.display = 'none';
            // Recargar página para mostrar todos los proyectos
            window.location.reload();
        }
    });
    
    // Inicializar tooltips personalizados
    initCustomTooltips();
    
    // Inicializar event listeners para menús de lista
    document.querySelectorAll('.dropdown-menu-list .menu-item-list').forEach(item => {
        item.addEventListener('click', function() {
            setTimeout(() => {
                const allMenus = document.querySelectorAll('.dropdown-menu-list');
                allMenus.forEach(menu => {
                    menu.style.display = 'none';
                    menu.style.position = 'fixed';
                });
            }, 100);
        });
    });
});

// Función para inicializar tooltips personalizados
function initCustomTooltips() {
    // Crear el elemento tooltip si no existe
    let tooltip = document.getElementById('custom-tooltip');
    if (!tooltip) {
        tooltip = document.createElement('div');
        tooltip.id = 'custom-tooltip';
        tooltip.className = 'custom-tooltip';
        document.body.appendChild(tooltip);
    }
    
    // Agregar event listeners a todos los elementos con tooltip
    document.addEventListener('mouseenter', function(e) {
        if (e.target.classList.contains('project-tooltip') && e.target.dataset.tooltip) {
            showCustomTooltip(e.target, e.target.dataset.tooltip);
        }
    }, true);
    
    document.addEventListener('mouseleave', function(e) {
        if (e.target.classList.contains('project-tooltip')) {
            hideCustomTooltip();
        }
    }, true);
    
    document.addEventListener('mousemove', function(e) {
        if (document.getElementById('custom-tooltip').style.display === 'block') {
            updateTooltipPosition(e);
        }
    });
}

// Mostrar tooltip personalizado
function showCustomTooltip(element, text) {
    const tooltip = document.getElementById('custom-tooltip');
    tooltip.textContent = text;
    tooltip.style.display = 'block';
    tooltip.style.opacity = '1';
}

// Ocultar tooltip personalizado
function hideCustomTooltip() {
    const tooltip = document.getElementById('custom-tooltip');
    tooltip.style.opacity = '0';
    setTimeout(() => {
        tooltip.style.display = 'none';
    }, 200);
}

// Actualizar posición del tooltip
function updateTooltipPosition(e) {
    const tooltip = document.getElementById('custom-tooltip');
    const tooltipRect = tooltip.getBoundingClientRect();
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;
    
    let left = e.clientX + 15;
    let top = e.clientY - tooltipRect.height - 15;
    
    // Ajustar si se sale por la derecha
    if (left + tooltipRect.width > viewportWidth - 10) {
        left = e.clientX - tooltipRect.width - 15;
    }
    
    // Ajustar si se sale por la izquierda
    if (left < 10) {
        left = 10;
    }
    
    // Ajustar si se sale por arriba
    if (top < 10) {
        top = e.clientY + 15;
    }
    
    // Ajustar si se sale por abajo
    if (top + tooltipRect.height > viewportHeight - 10) {
        top = viewportHeight - tooltipRect.height - 10;
    }
    
    tooltip.style.left = left + 'px';
    tooltip.style.top = top + 'px';
}
</script>

<style>
/* Header compacto */
.minimal-header {
    padding: 14px 16px !important;
    border-radius: 10px !important;
}

.minimal-header h1 {
    font-size: 20px !important;
}

.minimal-header .subtitle {
    font-size: 12px !important;
}

.clan-icon-large {
    font-size: 28px !important;
}

.btn-minimal {
    padding: 8px 14px !important;
    font-size: 13px !important;
}

.search-minimal {
    margin-top: 12px !important;
}

/* Estilos para checkbox personalizado */
.checkbox-label {
    display: flex;
    align-items: center;
    cursor: pointer;
    gap: 10px;
    padding: 10px 0;
}

.checkbox-label input[type="checkbox"] {
    display: none;
}

.checkmark {
    width: 20px;
    height: 20px;
    background-color: #fff;
    border: 2px solid #d1d5db;
    border-radius: 4px;
    position: relative;
    transition: all 0.3s ease;
}

.checkbox-label:hover .checkmark {
    border-color: #3b82f6;
}

.checkbox-label input[type="checkbox"]:checked + .checkmark {
    background-color: #3b82f6;
    border-color: #3b82f6;
}

.checkbox-label input[type="checkbox"]:checked + .checkmark::after {
    content: '';
    position: absolute;
    left: 6px;
    top: 2px;
    width: 6px;
    height: 12px;
    border: solid white;
    border-width: 0 2px 2px 0;
    transform: rotate(45deg);
}

.checkbox-label i {
    color: #6b7280;
    margin-right: 5px;
}

.checkbox-label input[type="checkbox"]:checked ~ i {
    color: #3b82f6;
}

/* ========== ESTILOS CONSISTENTES PARA PROYECTOS ========== */

/* Toggle de vista */
.view-toggle-minimal {
    display: flex;
    background: rgba(255, 255, 255, 0.9);
    border-radius: 12px;
    padding: 4px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    margin-right: 16px;
    border: 1px solid rgba(0, 0, 0, 0.05);
}

.view-btn {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: none;
    background: transparent;
    color: #7f8c8d;
    cursor: pointer;
    border-radius: 8px;
    transition: all 0.3s ease;
    font-size: 13px;
    font-weight: 600;
}

.view-btn:hover {
    color: #2c3e50;
    background: rgba(116, 75, 162, 0.1);
}

.view-btn.active {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    box-shadow: 0 2px 8px rgba(102, 126, 234, 0.3);
}

.view-btn.active i {
    color: white;
}

/* Grid de proyectos minimalista */
.projects-grid-minimal {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 14px;
    padding: 12px 0;
}

/* Cards de proyecto consistentes */
.project-item-enhanced {
    background: white;
    border-radius: 10px;
    box-shadow: 0 3px 12px rgba(0, 0, 0, 0.06);
    padding: 14px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(0, 0, 0, 0.05);
    position: relative;
    overflow: hidden;
}

.project-item-enhanced::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(102, 126, 234, 0.03) 0%, rgba(118, 75, 162, 0.01) 100%);
    pointer-events: none;
}

.project-item-enhanced:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.10);
}

/* Header del proyecto minimalista */
.project-info-header {
    display: flex;
    align-items: flex-start;
    gap: 10px;
    margin-bottom: 12px;
    position: relative;
    z-index: 1;
}

.project-icon-minimal {
    flex-shrink: 0;
    width: 36px;
    height: 36px;
    border-radius: 8px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    color: white !important;
    box-shadow: 0 2px 8px rgba(102, 126, 234, 0.25);
}

.project-icon-minimal i {
    color: white !important;
}

.project-details-minimal {
    flex: 1;
    min-width: 0;
}

.project-name-minimal {
    font-size: 15px;
    font-weight: 700;
    color: #2c3e50;
    margin: 0 0 6px 0;
    line-height: 1.3;
}

.project-description-minimal {
    font-size: 12px;
    color: #7f8c8d;
    line-height: 1.4;
    margin-bottom: 10px;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}

.project-meta-minimal {
    display: flex;
    gap: 8px;
    align-items: center;
    flex-wrap: wrap;
}

.project-status-minimal {
    display: inline-block;
    padding: 3px 10px;
    border-radius: 14px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}

.project-status-minimal.status-active { 
    background: rgba(39, 174, 96, 0.1); 
    color: #27ae60; 
    border: 1px solid rgba(39, 174, 96, 0.2);
}

.project-status-minimal.status-completed { 
    background: rgba(52, 152, 219, 0.1); 
    color: #3498db; 
    border: 1px solid rgba(52, 152, 219, 0.2);
}

.project-status-minimal.status-pending { 
    background: rgba(243, 156, 18, 0.1); 
    color: #f39c12; 
    border: 1px solid rgba(243, 156, 18, 0.2);
}

.project-kpi-minimal {
    display: inline-flex;
    align-items: center;
    gap: 3px;
    font-size: 10px;
    color: #f39c12;
    font-weight: 600;
}

/* Menú minimalista */
.project-menu-minimal {
    position: relative;
}

.btn-menu-minimal {
    background: rgba(116, 75, 162, 0.1);
    border: none;
    padding: 8px;
    border-radius: 8px;
    cursor: pointer;
    color: #764ba2;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-menu-minimal:hover {
    background: rgba(116, 75, 162, 0.2);
    transform: scale(1.05);
}

.dropdown-menu-minimal {
    position: absolute;
    top: 100%;
    right: 0;
    background: white;
    border-radius: 12px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
    padding: 8px 0;
    min-width: 160px;
    z-index: 1000;
    border: 1px solid rgba(0, 0, 0, 0.05);
    display: none;
}

.dropdown-menu-minimal.show {
    display: block !important;
}

.menu-item-minimal {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 16px;
    background: none;
    border: none;
    width: 100%;
    text-align: left;
    cursor: pointer;
    font-size: 14px;
    color: #2c3e50;
    transition: all 0.2s ease;
    font-weight: 500;
}

.menu-item-minimal:hover {
    background: linear-gradient(90deg, rgba(102, 126, 234, 0.05) 0%, rgba(118, 75, 162, 0.05) 100%);
}

.menu-item-minimal.danger {
    color: #e74c3c;
}

.menu-item-minimal.danger:hover {
    background: rgba(231, 76, 60, 0.05);
}

/* Estadísticas minimalistas */
.project-stats-minimal {
    display: flex;
    gap: 10px;
    margin-bottom: 12px;
    justify-content: space-between;
}

.stat-item-minimal {
    text-align: center;
    padding: 8px;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 8px;
    flex: 1;
    transition: all 0.2s ease;
    border: 1px solid rgba(0, 0, 0, 0.05);
}

.stat-item-minimal:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
}

.stat-item-minimal.completed {
    background: linear-gradient(135deg, rgba(39, 174, 96, 0.1) 0%, rgba(39, 174, 96, 0.05) 100%);
    border-color: rgba(39, 174, 96, 0.2);
}

.stat-item-minimal.progress {
    background: linear-gradient(135deg, rgba(52, 152, 219, 0.1) 0%, rgba(52, 152, 219, 0.05) 100%);
    border-color: rgba(52, 152, 219, 0.2);
}

.stat-icon-minimal {
    font-size: 14px;
    margin-bottom: 4px;
    color: #7f8c8d;
    transition: all 0.2s ease;
}

.stat-item-minimal.completed .stat-icon-minimal {
    color: #27ae60;
}

.stat-item-minimal.progress .stat-icon-minimal {
    color: #3498db;
}

.stat-number-minimal {
    font-size: 16px;
    font-weight: 700;
    color: #2c3e50;
    line-height: 1;
    margin-bottom: 3px;
}

.stat-label-minimal {
    font-size: 9px;
    color: #7f8c8d;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-weight: 600;
}

/* Barra de progreso minimalista */
.progress-bar-minimal {
    height: 6px;
    background: rgba(236, 240, 241, 0.8);
    border-radius: 3px;
    overflow: hidden;
    margin-bottom: 12px;
    position: relative;
}

.progress-fill-minimal {
    height: 100%;
    background: linear-gradient(90deg, #27ae60 0%, #2ecc71 100%);
    border-radius: 3px;
    transition: width 0.6s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
}

.progress-fill-minimal::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.3) 50%, transparent 100%);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

/* Sección de delegación minimalista */
.delegation-section-minimal {
    margin-bottom: 16px;
}

.checkbox-label-minimal {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    padding: 8px 0;
    transition: all 0.2s ease;
}

.checkbox-label-minimal input[type="checkbox"] {
    display: none !important;
    position: absolute !important;
    left: -9999px !important;
    opacity: 0 !important;
    pointer-events: none !important;
}

/* Asegurar que no aparezcan checkboxes nativos */
.delegation-toggle-hidden {
    display: none !important;
    position: absolute !important;
    left: -9999px !important;
    opacity: 0 !important;
    pointer-events: none !important;
}

.checkbox-label-minimal:hover {
    opacity: 0.8;
}

.checkbox-label-minimal:hover .checkmark-custom {
    border-color: #764ba2;
    transform: scale(1.05);
}

.checkbox-label-minimal .checkmark-custom {
    width: 24px;
    height: 24px;
    background-color: #fff;
    border: 2px solid #d1d5db;
    border-radius: 6px;
    position: relative;
    transition: all 0.3s ease;
    flex-shrink: 0;
    cursor: pointer;
}

.checkbox-label-minimal input[type="checkbox"]:checked + .checkmark-custom {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-color: #667eea;
    box-shadow: 0 2px 8px rgba(102, 126, 234, 0.3);
}

.checkbox-label-minimal input[type="checkbox"]:checked + .checkmark-custom::after {
    content: '';
    position: absolute;
    left: 8px;
    top: 3px;
    width: 6px;
    height: 12px;
    border: solid white;
    border-width: 0 2px 2px 0;
    transform: rotate(45deg);
}

.checkbox-text {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 14px;
    color: #5a6c7d;
    font-weight: 500;
}

.checkbox-text i {
    color: #764ba2;
    font-size: 16px;
}

/* Acciones minimalistas - 2 botones */
.project-actions-minimal {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.project-actions-minimal .btn-minimal {
    flex: 1;
    min-width: 0;
    justify-content: center;
    padding: 12px 16px;
    font-size: 14px;
    font-weight: 600;
}

/* Estado vacío minimalista */
.empty-minimal {
    text-align: center;
    padding: 80px 20px;
    color: #7f8c8d;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 20px;
    margin: 40px 0;
    border: 2px dashed rgba(127, 140, 141, 0.2);
}

.empty-icon-minimal {
    font-size: 64px;
    margin-bottom: 20px;
    opacity: 0.6;
}

.empty-minimal h3 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-weight: 600;
}

.empty-minimal p {
    margin: 0 0 30px 0;
    font-size: 16px;
    line-height: 1.5;
}

/* ========== ESTILOS PARA VISTA DE LISTA ========== */

/* Tabla minimalista */
.projects-table-minimal {
    background: white;
    border-radius: 16px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.08);
    overflow: visible;
    border: 1px solid rgba(0, 0, 0, 0.05);
}

.table-header-minimal {
    display: grid;
    grid-template-columns: 2fr 1fr 1fr 1fr 1.2fr;
    gap: 16px;
    padding: 16px 20px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    font-weight: 600;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.header-cell {
    display: flex;
    align-items: center;
}

.table-row-minimal {
    display: grid;
    grid-template-columns: 2fr 1fr 1fr 1fr 1.2fr;
    gap: 16px;
    padding: 16px 20px;
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
    transition: all 0.2s ease;
    align-items: center;
    overflow: visible !important;
    position: relative;
}

.table-row-minimal:hover {
    background: linear-gradient(90deg, rgba(102, 126, 234, 0.02) 0%, rgba(118, 75, 162, 0.02) 100%);
}

.table-row-minimal:last-child {
    border-bottom: none;
}

/* Celda de proyecto */
.cell-project {
    display: flex;
    align-items: center;
    gap: 12px;
}

.project-icon-list {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 16px;
    flex-shrink: 0;
}

.project-info-list {
    flex: 1;
    min-width: 0;
}

.project-name-list {
    font-size: 15px;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 4px;
    line-height: 1.3;
}

.project-description-list {
    font-size: 12px;
    color: #7f8c8d;
    line-height: 1.3;
    display: -webkit-box;
    -webkit-line-clamp: 1;
    -webkit-box-orient: vertical;
    overflow: hidden;
    margin-bottom: 4px;
}

.project-kpi-list {
    font-size: 11px;
    color: #f39c12;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
}

/* Celda de estado */
.cell-status {
    display: flex;
    justify-content: center;
}

.status-badge-list {
    padding: 4px 12px;
    border-radius: 16px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}

.status-badge-list.status-active { 
    background: rgba(39, 174, 96, 0.1); 
    color: #27ae60; 
    border: 1px solid rgba(39, 174, 96, 0.2);
}

.status-badge-list.status-completed { 
    background: rgba(52, 152, 219, 0.1); 
    color: #3498db; 
    border: 1px solid rgba(52, 152, 219, 0.2);
}

.status-badge-list.status-pending { 
    background: rgba(243, 156, 18, 0.1); 
    color: #f39c12; 
    border: 1px solid rgba(243, 156, 18, 0.2);
}

/* Celda de progreso */
.cell-progress {
    display: flex;
    justify-content: center;
}

.progress-container-list {
    display: flex;
    align-items: center;
    gap: 8px;
    width: 100%;
}

.progress-bar-list {
    flex: 1;
    height: 6px;
    background: #ecf0f1;
    border-radius: 3px;
    overflow: hidden;
}

.progress-fill-list {
    height: 100%;
    background: linear-gradient(90deg, #27ae60 0%, #2ecc71 100%);
    border-radius: 3px;
    transition: width 0.3s ease;
}

.progress-text-list {
    font-size: 12px;
    font-weight: 600;
    color: #2c3e50;
    min-width: 35px;
}

/* Celda de tareas */
.cell-tasks {
    display: flex;
    justify-content: center;
}

.tasks-summary-list {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 14px;
    font-weight: 600;
}

.tasks-total {
    color: #2c3e50;
}

.tasks-separator {
    color: #bdc3c7;
}

.tasks-completed {
    color: #27ae60;
}

/* Celda de acciones */
.cell-actions {
    display: flex;
    justify-content: center;
    position: relative;
}

.actions-list {
    display: flex;
    gap: 6px;
    position: relative;
}

.btn-list-action {
    padding: 8px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 12px;
    transition: all 0.2s ease;
    text-decoration: none;
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
}

.btn-list-action.primary {
    background: #669AE1;
    color: white;
}

.btn-list-action.primary:hover {
    background: #4a7bc8;
    transform: scale(1.05);
}

.btn-list-action.secondary {
    background: #669AE1;
    color: white;
}

.btn-list-action.secondary:hover {
    background: #4a7bc8;
    transform: scale(1.05);
}

.btn-list-action.menu {
    background: rgba(116, 75, 162, 0.1);
    color: #764ba2;
}

.btn-list-action.menu:hover {
    background: rgba(116, 75, 162, 0.2);
    transform: scale(1.05);
}

/* Contenedor del dropdown para lista */


/* Estilos para búsqueda en tiempo real */
.search-input {
    position: relative;
    display: flex;
    align-items: center;
}

.search-input input {
    padding-right: 40px;
    transition: padding-right 0.3s ease;
}

.search-input .btn-clear {
    position: absolute;
    right: 10px;
    background: none;
    border: none;
    color: #7f8c8d;
    cursor: pointer;
    padding: 5px;
    border-radius: 4px;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    width: 24px;
    height: 24px;
}

.search-input .btn-clear:hover {
    background: rgba(231, 76, 60, 0.1);
    color: #e74c3c;
}

.search-loading {
    position: absolute;
    right: 45px;
    color: #667eea;
    font-size: 14px;
}

/* Animación para el loading */
.search-loading i {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Mejorar el estilo del campo de búsqueda */
.search-minimal .search-input {
    position: relative;
    flex: 1;
    max-width: 400px;
}

.search-minimal .search-input input {
    width: 100%;
    padding: 12px 16px;
    padding-right: 40px;
    border: 2px solid rgba(102, 126, 234, 0.2);
    border-radius: 12px;
    background: rgba(255, 255, 255, 0.9);
    font-size: 14px;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
}

.search-minimal .search-input input:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    background: white;
}

.search-minimal .search-input i.fa-search {
    position: absolute;
    left: 12px;
    color: #7f8c8d;
    font-size: 14px;
    pointer-events: none;
}

/* Responsive design */
@media (max-width: 768px) {
    .projects-grid-minimal {
        grid-template-columns: 1fr;
        gap: 16px;
        padding: 16px 0;
    }
    
    .project-stats-minimal {
        flex-direction: column;
        gap: 8px;
    }
    
    .stat-item-minimal {
        padding: 8px;
    }
    
    .project-actions-minimal {
        flex-direction: column;
    }
    
    .project-actions-minimal .btn-minimal {
        flex: none;
    }
    
    /* Vista de lista responsive */
    .table-header-minimal,
    .table-row-minimal {
        grid-template-columns: 1fr;
        gap: 8px;
    }
    
    .table-header-minimal {
        display: none; /* Ocultar header en móvil */
    }
    
    .table-row-minimal {
        padding: 16px;
        border-radius: 12px;
        margin-bottom: 12px;
        background: white;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        border: 1px solid rgba(0, 0, 0, 0.05);
    }
    
    .cell-project,
    .cell-status,
    .cell-progress,
    .cell-tasks,
    .cell-actions {
        justify-content: flex-start;
    }
    
    .view-toggle-minimal {
        margin-bottom: 12px;
        margin-right: 0;
    }
}

.project-menu-btn {
    background: none;
    border: none;
    padding: 8px;
    border-radius: 6px;
    cursor: pointer;
    color: #666;
    transition: all 0.2s ease;
}

.project-menu-btn:hover {
    background: #f5f5f5;
    color: #333;
}

.project-menu {
    position: absolute;
    top: 100%;
    right: 0;
    background: white;
    border-radius: 8px;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    padding: 8px 0;
    min-width: 180px;
    z-index: 1000;
    opacity: 0;
    visibility: hidden;
    transform: translateY(-10px);
    transition: all 0.3s ease;
}

.project-menu.show {
    opacity: 1;
    visibility: visible;
    transform: translateY(0);
}

.menu-item {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 10px 16px;
    background: none;
    border: none;
    width: 100%;
    text-align: left;
    cursor: pointer;
    font-size: 14px;
    color: #333;
    transition: background 0.2s ease;
}

.menu-item:hover {
    background: #f8f9fa;
}



/* Acciones del proyecto */
.project-actions-enhanced {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.btn-action {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: none;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 600;
    text-decoration: none;
    cursor: pointer;
    transition: all 0.2s ease;
    flex: 1;
    justify-content: center;
    min-width: 0;
}

.btn-action.primary {
    background: #007bff;
    color: white;
}

.btn-action.primary:hover {
    background: #0056b3;
    transform: translateY(-1px);
}

.btn-action.secondary {
    background: #6c757d;
    color: white;
}

.btn-action.secondary:hover {
    background: #545b62;
    transform: translateY(-1px);
}

.btn-action.info {
    background: #17a2b8;
    color: white;
}

.btn-action.info:hover {
    background: #138496;
    transform: translateY(-1px);
}

/* Sección de tareas del proyecto */
.project-tasks-section {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 2px solid #f1f3f4;
    background: #fafbfc;
    border-radius: 8px;
    padding: 15px;
    animation: slideDown 0.3s ease;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.tasks-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.tasks-header h4 {
    margin: 0;
    color: #2c3e50;
    font-size: 16px;
}

.btn-mini {
    padding: 6px 10px;
    border: none;
    border-radius: 4px;
    background: #e9ecef;
    color: #495057;
    cursor: pointer;
    font-size: 12px;
    transition: all 0.2s ease;
}

.btn-mini:hover {
    background: #dee2e6;
}

.btn-mini.primary {
    background: #007bff;
    color: white;
}

.btn-mini.primary:hover {
    background: #0056b3;
}

/* Lista de tareas */
.tasks-list {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.task-item-mini {
    background: white;
    border-radius: 8px;
    padding: 12px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
    transition: all 0.2s ease;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    border-left: 4px solid #dee2e6;
}

.task-item-mini:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
}

.task-item-mini.completed {
    border-left-color: #27ae60;
    opacity: 0.8;
}

.task-item-mini.in-progress {
    border-left-color: #3498db;
}

.task-item-mini.pending {
    border-left-color: #f39c12;
}

.task-info {
    flex: 1;
}

.task-header-inline {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 8px;
}

.task-title {
    font-size: 14px;
    font-weight: 600;
    color: #2c3e50;
    margin: 0;
    line-height: 1.3;
}

.task-badges {
    display: flex;
    gap: 6px;
}

.task-status, .task-priority {
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.task-status.status-completed { background: #d4edda; color: #155724; }
.task-status.status-in-progress { background: #cce7ff; color: #004085; }
.task-status.status-pending { background: #fff3cd; color: #856404; }

.task-priority.priority-high { background: #f8d7da; color: #721c24; }
.task-priority.priority-medium { background: #fff3cd; color: #856404; }
.task-priority.priority-low { background: #d1ecf1; color: #0c5460; }

.task-description {
    font-size: 12px;
    color: #666;
    margin: 0 0 8px 0;
    line-height: 1.4;
}

.task-meta {
    display: flex;
    gap: 12px;
    font-size: 11px;
    color: #666;
}

.task-meta span {
    display: flex;
    align-items: center;
    gap: 4px;
}

/* Acciones de tarea */
.task-actions-mini {
    display: flex;
    gap: 4px;
    flex-shrink: 0;
}

.btn-task-action {
    padding: 6px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 12px;
    transition: all 0.2s ease;
    text-decoration: none;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-task-action.edit {
    background: #fff3cd;
    color: #856404;
}

.btn-task-action.edit:hover {
    background: #ffeaa7;
}

.btn-task-action.delete {
    background: #f8d7da;
    color: #721c24;
}

.btn-task-action.delete:hover {
    background: #f5c6cb;
}

.btn-task-action.view {
    background: #cce7ff;
    color: #004085;
}

.btn-task-action.view:hover {
    background: #b3d7ff;
}

/* Estados especiales */
.loading-tasks, .no-tasks, .error-tasks {
    text-align: center;
    padding: 30px;
    color: #666;
}

.loading-tasks i {
    font-size: 24px;
    color: #007bff;
    margin-bottom: 10px;
}

.no-tasks i, .error-tasks i {
    font-size: 32px;
    margin-bottom: 15px;
    opacity: 0.5;
}

/* Toast notifications */
.toast {
    position: fixed;
    top: 20px;
    right: 20px;
    padding: 12px 20px;
    border-radius: 6px;
    color: white;
    font-weight: 600;
    z-index: 10000;
    opacity: 0;
    transform: translateX(100%);
    transition: all 0.3s ease;
}

.toast.show {
    opacity: 1;
    transform: translateX(0);
}

.toast-success { background: #27ae60; }
.toast-error { background: #e74c3c; }
.toast-info { background: #3498db; }

/* Estilos para el modal de clonación de proyectos */
.clone-project-modal {
    max-width: 700px;
    width: 90%;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #374151;
}

.form-group input,
.form-group textarea {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    font-size: 14px;
    transition: border-color 0.2s ease;
}

.form-group input:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-group input[type="checkbox"] {
    width: auto;
    margin-right: 0.5rem;
}

.form-help {
    display: block;
    margin-top: 0.25rem;
    font-size: 12px;
    color: #6b7280;
    font-style: italic;
}

/* Estilos del modal iguales al de miembros */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    z-index: 1000;
    backdrop-filter: blur(4px);
}

.modal-content {
    background: white;
    border-radius: 12px;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
    width: 90%;
    max-width: 600px;
    max-height: 80vh;
    min-height: 400px;
    overflow: hidden;
    animation: modalFadeIn 0.3s ease-out;
    margin: 10% auto;
    position: relative;
}

@keyframes modalFadeIn {
    from {
        opacity: 0;
        transform: scale(0.9) translateY(-20px);
    }
    to {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

.modal-header {
    padding: 20px 24px;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    align-items: center;
    justify-content: space-between;
    background: white;
    color: #374151;
}

.modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 8px;
}

.modal-close {
    background: none;
    border: none;
    color: #6b7280;
    font-size: 18px;
    cursor: pointer;
    padding: 8px;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.modal-close:hover {
    background: #f3f4f6;
    color: #374151;
    transform: scale(1.1);
}

.modal-body {
    padding: 24px;
    max-height: 50vh;
    min-height: 250px;
    overflow-y: auto;
}

.modal-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 500;
    color: #374151;
    display: flex;
    align-items: center;
    gap: 8px;
}

.form-group input,
.form-group textarea {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.2s ease, box-shadow 0.2s ease;
    background: #fff;
}

.form-group input:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #1e3a8a;
    box-shadow: 0 0 0 3px rgba(30, 58, 138, 0.1);
}

.modal-footer {
    padding: 20px 24px;
    border-top: 1px solid #e2e8f0;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
    background: #f9fafb;
}

/* Estilos de botones iguales al modal de miembros */
.btn-primary,
.btn-secondary {
    padding: 12px 24px;
    border-radius: 8px;
    font-weight: 600;
    font-size: 1rem;
    cursor: pointer;
    transition: all 0.2s ease;
    border: none;
    display: flex;
    align-items: center;
    gap: 8px;
}

.btn-primary {
    background: #1e3a8a;
    color: white;
}

.btn-primary:hover {
    background: #1e3a8a;
    transform: translateY(-2px);
    box-shadow: 0 8px 16px rgba(30, 58, 138, 0.3);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
}

.btn-secondary:hover {
    background: #e5e7eb;
    transform: translateY(-2px);
}

@media (max-width: 768px) {
    .clone-project-modal {
        width: 95%;
        margin: 1rem;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .modal-content {
        width: 95%;
        margin: 20px;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 16px 20px;
    }
    
    .modal-footer {
        flex-direction: column;
    }
    
    .btn-primary, .btn-secondary {
        justify-content: center;
        width: 100%;
    }
}

/* Estilos para modal de confirmación personalizado */
.confirmation-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 10000;
    opacity: 0;
    transition: opacity 0.3s ease;
    backdrop-filter: blur(4px);
}

.confirmation-modal-overlay.show {
    opacity: 1;
}

.confirmation-modal {
    background: white;
    border-radius: 12px;
    padding: 0;
    max-width: 400px;
    width: 90%;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    transform: scale(0.9);
    transition: transform 0.3s ease;
}

.confirmation-modal-overlay.show .confirmation-modal {
    transform: scale(1);
}

.confirmation-modal-header {
    padding: 20px 24px 0;
}

.confirmation-modal-title {
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
    display: flex;
    align-items: center;
    gap: 8px;
}

.confirmation-modal-body {
    padding: 20px 24px 24px;
    text-align: center;
}

.confirmation-modal-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
}

.confirmation-modal-icon.warning {
    color: #f59e0b;
}

.confirmation-modal-icon.danger {
    color: #ef4444;
}

.confirmation-modal-message {
    margin: 0 0 1.5rem 0;
    color: #6b7280;
    line-height: 1.5;
    white-space: pre-line;
}

.confirmation-modal-actions {
    display: flex;
    gap: 12px;
    justify-content: center;
}

.confirmation-modal-btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    min-width: 100px;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.confirmation-modal-btn.cancel {
    background: #f3f4f6;
    color: #374151;
}

.confirmation-modal-btn.cancel:hover {
    background: #e5e7eb;
    transform: translateY(-1px);
}

.confirmation-modal-btn.confirm {
    background: #ef4444;
    color: white;
}

.confirmation-modal-btn.confirm:hover {
    background: #dc2626;
    transform: translateY(-1px);
}

@media (max-width: 768px) {
    .confirmation-modal {
        width: 95%;
        margin: 20px;
    }
    
    .confirmation-modal-actions {
        flex-direction: column;
    }
    
    .confirmation-modal-btn {
        width: 100%;
    }
}
/* Estilos para el nuevo diseño con monday-theme */
.view-toggle-monday {
    display: flex;
    gap: 8px;
    margin-right: 12px;
}

.view-toggle-monday .view-btn.active {
    background: var(--monday-primary);
    color: white;
}

.search-container-monday {
    position: relative;
}

.search-input-monday {
    position: relative;
    display: flex;
    align-items: center;
    background: white;
    border: 1px solid var(--monday-border);
    border-radius: var(--border-radius);
    padding: 0 12px;
    transition: all 0.2s;
}

.search-input-monday:focus-within {
    border-color: var(--monday-primary);
    box-shadow: 0 0 0 3px rgba(30, 58, 138, 0.1);
}

.search-input-monday i {
    color: var(--monday-text-secondary);
    margin-right: 8px;
}

.search-input-monday input {
    flex: 1;
    border: none;
    outline: none;
    padding: 10px 0;
    font-size: 14px;
    color: var(--monday-text-color);
    background: transparent;
}

.search-input-monday input::placeholder {
    color: var(--monday-text-secondary);
}

.btn-clear-monday {
    background: none;
    border: none;
    color: var(--monday-text-secondary);
    cursor: pointer;
    padding: 4px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: all 0.2s;
}

.btn-clear-monday:hover {
    background: var(--monday-bg-secondary);
    color: var(--monday-text-color);
}

.search-loading-monday {
    position: absolute;
    right: 40px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--monday-primary);
}

/* Estilos para la vista de lista con monday-theme */
.projects-list-view {
    background: white;
    border-radius: var(--border-radius);
    box-shadow: var(--box-shadow);
    overflow: hidden;
}

.projects-table-minimal {
    width: 100%;
}

.table-header-minimal {
    display: grid;
    grid-template-columns: 2fr 1fr 1.5fr 1fr 1.5fr;
    gap: 16px;
    padding: 16px 20px;
    background: var(--monday-bg-secondary);
    border-bottom: 2px solid var(--monday-border);
    font-weight: 600;
    font-size: 13px;
    color: var(--monday-text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.table-header-minimal > div {
    display: flex;
    align-items: center;
    justify-content: center;
    text-align: center;
}

.table-header-minimal > div:first-child {
    justify-content: flex-start;
    text-align: left;
}

.table-row-minimal {
    display: grid;
    grid-template-columns: 2fr 1fr 1.5fr 1fr 1.5fr;
    gap: 16px;
    padding: 16px 20px;
    border-bottom: 1px solid var(--monday-border);
    transition: background 0.2s;
    align-items: center;
}

.table-row-minimal:hover {
    background: var(--monday-bg-secondary);
}

.cell-project {
    display: flex;
    align-items: center;
    gap: 12px;
}

.project-icon-list {
    width: 40px;
    height: 40px;
    background: var(--monday-primary);
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 18px;
    flex-shrink: 0;
}

.project-info-list {
    flex: 1;
    min-width: 0;
}

.project-name-list {
    font-weight: 600;
    color: var(--monday-text-color);
    margin-bottom: 4px;
    font-size: 14px;
}

.project-description-list {
    font-size: 12px;
    color: var(--monday-text-secondary);
    margin-bottom: 4px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.project-kpi-list {
    font-size: 11px;
    color: var(--monday-primary);
    display: flex;
    align-items: center;
    gap: 4px;
    margin-top: 4px;
}

.cell-status {
    display: flex;
    align-items: center;
    justify-content: center;
}

.status-badge-list {
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-badge-list.status-active {
    background: rgba(16, 185, 129, 0.1);
    color: #10b981;
}

.status-badge-list.status-pending {
    background: rgba(245, 158, 11, 0.1);
    color: #f59e0b;
}

.status-badge-list.status-completed {
    background: rgba(59, 130, 246, 0.1);
    color: #3b82f6;
}

.cell-progress {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
}

.progress-container-list {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
    width: 100%;
}

.progress-bar-list {
    width: 80px;
    height: 8px;
    background: var(--monday-bg-secondary);
    border-radius: 4px;
    overflow: hidden;
}

.progress-fill-list {
    height: 100%;
    background: linear-gradient(90deg, var(--monday-primary), #3b82f6);
    border-radius: 4px;
    transition: width 0.3s;
}

.progress-text-list {
    font-size: 12px;
    font-weight: 600;
    color: var(--monday-text-color);
    min-width: 40px;
    text-align: right;
}

.cell-tasks {
    display: flex;
    align-items: center;
    justify-content: center;
}

.tasks-summary-list {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 4px;
    font-size: 13px;
    font-weight: 600;
}

.tasks-total {
    color: var(--monday-text-color);
}

.tasks-separator {
    color: var(--monday-text-secondary);
}

.tasks-completed {
    color: var(--monday-primary);
}

.cell-actions {
    display: flex;
    align-items: center;
    justify-content: center;
    overflow: visible !important;
    position: relative;
}

.actions-list {
    display: flex;
    gap: 8px;
    align-items: center;
    overflow: visible !important;
    position: relative;
}

.btn-list-action {
    width: 32px;
    height: 32px;
    border: none;
    border-radius: 6px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s;
    font-size: 14px;
}

.btn-list-action.primary {
    background: #669AE1;
    color: white;
}

.btn-list-action.primary:hover {
    background: #4a7bc8;
    transform: scale(1.05);
}

.btn-list-action.secondary {
    background: #669AE1;
    color: white;
}

.btn-list-action.secondary:hover {
    background: #4a7bc8;
    transform: scale(1.05);
}

.btn-list-action.menu {
    background: transparent;
    color: var(--monday-text-secondary);
}

.btn-list-action.menu:hover {
    background: var(--monday-bg-secondary);
    color: var(--monday-text-color);
}

/* Dropdown Menu para Lista */
.dropdown-container {
    position: relative;
    display: inline-block;
    overflow: visible !important;
    z-index: 1000;
}

.btn-list-action.menu {
    background: #669AE1;
    color: white;
    border: none;
    width: 32px;
    height: 32px;
    border-radius: 6px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 14px;
    transition: all 0.2s ease;
}

.btn-list-action.menu:hover {
    background: #4a7bc8;
    transform: scale(1.05);
}

.dropdown-menu-list {
    display: none;
    position: fixed;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    box-shadow: 0 12px 40px rgba(0, 0, 0, 0.25);
    z-index: 999999;
    min-width: 180px;
    max-height: 300px;
    overflow-y: auto;
    padding: 6px 0;
}

.menu-item-list {
    display: flex;
    align-items: center;
    gap: 8px;
    width: 100%;
    padding: 12px 16px;
    border: none;
    background: none;
    color: #374151;
    font-size: 14px;
    cursor: pointer;
    transition: background-color 0.2s ease;
    text-align: left;
}

.menu-item-list:hover {
    background-color: #f8fafc;
}

.menu-item-list.danger {
    color: #dc2626;
}

.menu-item-list.danger:hover {
    background-color: #fef2f2;
}

.menu-item-list i {
    width: 16px;
    text-align: center;
}

/* Estilos para paginación */
.pagination-container-monday {
    padding: 20px;
    border-top: 1px solid #e2e8f0;
    background: #f8fafc;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 16px;
}

.pagination-info-monday {
    font-size: 13px;
    color: #64748b;
    font-weight: 500;
}

.pagination-controls-monday {
    display: flex;
    align-items: center;
    gap: 8px;
}

.pagination-pages-monday {
    display: flex;
    align-items: center;
    gap: 4px;
}

.btn-pagination-monday {
    padding: 8px 16px;
    border: 1px solid #e2e8f0;
    background: white;
    color: #374151;
    border-radius: 6px;
    cursor: pointer;
    font-size: 13px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 6px;
    transition: all 0.2s;
    text-decoration: none;
}

.btn-pagination-monday:hover:not(:disabled) {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.btn-pagination-monday:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.btn-page-monday {
    min-width: 36px;
    height: 36px;
    padding: 0 8px;
    border: 1px solid #e2e8f0;
    background: white;
    color: #374151;
    border-radius: 6px;
    cursor: pointer;
    font-size: 13px;
    font-weight: 500;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s;
}

.btn-page-monday:hover {
    background: #f1f5f9;
    border-color: #3b82f6;
}

.btn-page-monday.active {
    background: #3b82f6;
    color: white;
    border-color: #3b82f6;
}

.pagination-ellipsis {
    padding: 0 8px;
    color: #64748b;
    font-size: 13px;
}

/* Responsive para paginación */
@media (max-width: 768px) {
    .pagination-container-monday {
        flex-direction: column;
        gap: 12px;
        padding: 16px;
    }
    
    .pagination-controls-monday {
        width: 100%;
        justify-content: center;
    }
    
    .pagination-pages-monday {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .btn-pagination-monday,
    .btn-page-monday {
        font-size: 12px;
        padding: 6px 12px;
        min-width: 32px;
        height: 32px;
    }
}

/* Botón verde estilo Recordatorio */
.btn-monday.btn-primary.btn-green {
    background: linear-gradient(135deg, #059669 0%, #10b981 100%) !important;
    border: none !important;
    color: white !important;
    position: relative;
    overflow: hidden;
    z-index: 1;
}

.btn-monday.btn-primary.btn-green::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
    transition: left 0.5s;
}

.btn-monday.btn-primary.btn-green:hover::before {
    left: 100%;
}

.btn-monday.btn-primary.btn-green:hover {
    background: linear-gradient(135deg, #047857 0%, #059669 100%) !important;
    transform: translateY(-2px);
    box-shadow: 
        0 8px 20px rgba(16, 185, 129, 0.4),
        0 0 0 1px rgba(255, 255, 255, 0.2) inset;
}

.btn-monday.btn-primary.btn-green:active {
    transform: translateY(0);
}
</style>

<?php
// Guardar el contenido en una variable
error_log("Vista projects.php - ANTES de ob_get_clean - Longitud del buffer: " . ob_get_length());
$content = ob_get_clean();
error_log("Vista projects.php - DESPUÉS de ob_get_clean - Longitud del contenido: " . strlen($content));
error_log("Vista projects.php - ¿Contiene 'project-item-enhanced'?: " . (strpos($content, 'project-item-enhanced') !== false ? 'SÍ' : 'NO'));

// CSS adicional para Clan Leader
$additionalCSS = [
    APP_URL . 'assets/css/clan-leader.css'
];

// JavaScript adicional para Clan Leader  
$additionalJS = [
    APP_URL . 'assets/js/clan-leader.js'
];

// Incluir el layout
require_once __DIR__ . '/../admin/layout.php';
?> 