<?php

class NotificationService {
    private $mailer;

    public function __construct() {
        $this->mailer = new Mailer();
    }

    public function notifyTaskAssigned($taskId, $assignedUserIds) {
        $setting = Notification::getSetting('task_assigned');
        if (!(int)($setting['is_enabled'] ?? 0)) return 0;

        if (empty($assignedUserIds)) return 0;
        $assignedUserIds = array_values(array_unique(array_map('intval', $assignedUserIds)));
        $db = Database::getConnection();
        // Obtener info de tarea/proyecto
        $taskStmt = $db->prepare("SELECT t.task_id, t.task_name, t.due_date, t.project_id, p.project_name FROM Tasks t JOIN Projects p ON t.project_id = p.project_id WHERE t.task_id = ?");
        $taskStmt->execute([$taskId]);
        $taskInfo = $taskStmt->fetch();
        if (!$taskInfo) { return 0; }

        // Obtener usuarios destino con email
        $placeholders = implode(',', array_fill(0, count($assignedUserIds), '?'));
        $userStmt = $db->prepare("SELECT user_id, full_name, email FROM Users WHERE user_id IN ($placeholders) AND email IS NOT NULL AND email != '' AND is_active = 1");
        $userStmt->execute($assignedUserIds);
        $rows = $userStmt->fetchAll();
        $count = 0;
        foreach ($rows as $r) {
            $to = $r['email'];
            if (Notification::alreadySent('task_assigned', $taskId, $r['user_id'], $to)) continue;
            $html = EmailTemplate::render(
                'Nueva tarea asignada',
                '<p>Se te ha asignado una nueva tarea ✅.</p>',
                [
                    ['label' => 'Tarea', 'value' => $taskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $taskInfo['project_name']],
                    ['label' => 'Fecha de entrega', 'value' => ($taskInfo['due_date'] ?? null) ?: 'No definida']
                ],
                null,
                ['emoji' => '🧩']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Tarea asignada', $html)) {
                Notification::logSent('task_assigned', $taskId, $r['user_id'], $to);
                $count++;
            } else {
                error_log('task_assigned mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        // Extras globales para notificación
        $extra = $this->parseRecipients($setting['recipients'] ?? null);
        if (!empty($extra)) {
            if ($taskInfo) {
                $html = EmailTemplate::render(
                    'Tarea asignada (copia)',
                    '<p>Se ha asignado una tarea.</p>',
                    [
                        ['label' => 'Tarea', 'value' => $taskInfo['task_name']],
                        ['label' => 'Proyecto', 'value' => $taskInfo['project_name']]
                    ],
                    null,
                    ['emoji' => '🧩']
                );
                foreach ($extra as $mail) { $this->mailer->sendHtml($mail, 'Polaris • Tarea asignada', $html); }
            }
        }
        return $count;
    }

    public function notifyProjectAssignedToClan($projectId) {
        $setting = Notification::getSetting('project_assigned_to_clan');
        if (!(int)$setting['is_enabled']) return;

        $db = Database::getConnection();
        $stmt = $db->prepare("SELECT p.project_id, p.project_name, p.description, p.created_by_user_id, c.clan_id, c.clan_name, u.full_name as creator_name FROM Projects p JOIN Clans c ON p.clan_id = c.clan_id LEFT JOIN Users u ON p.created_by_user_id = u.user_id WHERE p.project_id = ?");
        $stmt->execute([$projectId]);
        $project = $stmt->fetch();
        if (!$project) return;

        // Miembros del clan con email
        $stmt = $db->prepare("SELECT u.user_id, u.full_name, u.email FROM Clan_Members cm JOIN Users u ON cm.user_id = u.user_id WHERE cm.clan_id = ? AND u.email IS NOT NULL AND u.email != '' AND u.is_active = 1");
        $stmt->execute([$project['clan_id']]);
        $members = $stmt->fetchAll();

        // Destinatarios adicionales desde configuración
        $extra = $this->parseRecipients($setting['recipients']);

        foreach ($members as $m) {
            $to = $m['email'];
            if (Notification::alreadySent('project_assigned_to_clan', $projectId, $m['user_id'], $to)) continue;
            $html = EmailTemplate::render(
                'Nuevo proyecto asignado',
                '<p>Se te ha asignado un nuevo proyecto en tu clan 🚀.</p>',
                [
                    ['label' => 'Proyecto', 'value' => $project['project_name']],
                    ['label' => 'Clan', 'value' => $project['clan_name']],
                    ['label' => 'Creado por', 'value' => $project['creator_name'] ?? ''],
                    ['label' => 'Descripción', 'value' => $project['description'] ?? '']
                ],
                null,
                ['emoji' => '📁']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Proyecto asignado', $html)) {
                Notification::logSent('project_assigned_to_clan', $projectId, $m['user_id'], $to);
            }
        }

        foreach ($extra as $to) {
            $html = EmailTemplate::render(
                'Proyecto asignado a un clan',
                '<p>Se ha creado/asignado un proyecto.</p>',
                [
                    ['label' => 'Proyecto', 'value' => $project['project_name']],
                    ['label' => 'Clan', 'value' => $project['clan_name']],
                    ['label' => 'Creado por', 'value' => $project['creator_name'] ?? '']
                ],
                ['label' => 'Ver proyecto', 'url' => APP_URL . '?route=admin/project-details&projectId=' . urlencode($projectId)]
            );
            $this->mailer->sendHtml($to, 'Polaris • Proyecto asignado', $html);
        }
    }

    public function notifyTaskDueSoonMulti() {
        $db = Database::getConnection();
        $total = 0;
        for ($i=1; $i<=3; $i++) {
            $key = 'task_due_soon_' . $i;
            $setting = Notification::getSetting($key);
            if (!(int)($setting['is_enabled'] ?? 0)) continue;
            $daysAhead = (int)($setting['value'] ?? 0);
            if ($daysAhead < 0 || $daysAhead > 365) continue;
            $sql = "SELECT t.task_id, t.task_name, t.due_date, t.project_id, p.project_name, t.assigned_to_user_id, u.full_name, u.email FROM Tasks t JOIN Projects p ON t.project_id = p.project_id LEFT JOIN Users u ON t.assigned_to_user_id = u.user_id WHERE t.status != 'completed' AND t.due_date IS NOT NULL AND t.due_date = DATE_ADD(CURDATE(), INTERVAL ? DAY) AND u.email IS NOT NULL AND u.email != ''";
            $stmt = $db->prepare($sql);
            $stmt->execute([$daysAhead]);
            $rows = $stmt->fetchAll();
            foreach ($rows as $r) {
                $to = $r['email'];
                if (Notification::alreadySent('task_due_soon_' . $daysAhead, $r['task_id'], $r['assigned_to_user_id'], $to)) continue;
            $html = EmailTemplate::render(
                'Recordatorio: tarea próxima a vencer',
                '<p>Tienes una tarea próxima a vencer ⏰.</p>',
                [
                    ['label' => 'Tarea', 'value' => $r['task_name']],
                    ['label' => 'Proyecto', 'value' => $r['project_name']],
                    ['label' => 'Vence', 'value' => $r['due_date']],
                    ['label' => 'Aviso', 'value' => 'En ' . $daysAhead . ' día(s)']
                ],
                null,
                ['emoji' => '⏳']
            );
                if ($this->mailer->sendHtml($to, 'Polaris • Tarea próxima a vencer (' . $daysAhead . ' días)', $html)) {
                    Notification::logSent('task_due_soon_' . $daysAhead, $r['task_id'], $r['assigned_to_user_id'], $to);
                    $total++;
                }
            }
        }
        return $total;
    }

    public function notifyTaskOverdue() {
        $setting = Notification::getSetting('task_overdue');
        if (!(int)$setting['is_enabled']) return 0;
        $db = Database::getConnection();
        $sql = "SELECT t.task_id, t.task_name, t.due_date, t.project_id, p.project_name, t.assigned_to_user_id, u.full_name, u.email FROM Tasks t JOIN Projects p ON t.project_id = p.project_id LEFT JOIN Users u ON t.assigned_to_user_id = u.user_id WHERE t.status != 'completed' AND t.due_date IS NOT NULL AND t.due_date < CURDATE() AND u.email IS NOT NULL AND u.email != ''";
        $rows = $db->query($sql)->fetchAll();
        $count = 0;
        foreach ($rows as $r) {
            $to = $r['email'];
            if (Notification::alreadySent('task_overdue', $r['task_id'], $r['assigned_to_user_id'], $to)) continue;
            $html = EmailTemplate::render(
                'Alerta: tarea vencida',
                '<p>Tienes una tarea vencida ⚠️. Por favor atiéndela cuanto antes.</p>',
                [
                    ['label' => 'Tarea', 'value' => $r['task_name']],
                    ['label' => 'Proyecto', 'value' => $r['project_name']],
                    ['label' => 'Venció', 'value' => $r['due_date']]
                ],
                null,
                ['emoji' => '⚠️']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Tarea vencida', $html)) {
                Notification::logSent('task_overdue', $r['task_id'], $r['assigned_to_user_id'], $to);
                $count++;
            }
        }
        return $count;
    }

    /**
     * Obtener todos los usuarios involucrados en una tarea (asignados, creador, participantes del proyecto)
     */
    private function getTaskInvolvedUsers($taskId, $excludeUserId = null) {
        $db = Database::getConnection();
        $users = [];
        
        // Obtener información de la tarea
        $taskStmt = $db->prepare("
            SELECT t.task_id, t.task_name, t.created_by_user_id, t.assigned_to_user_id, t.project_id, p.project_name, p.clan_id
            FROM Tasks t
            JOIN Projects p ON t.project_id = p.project_id
            WHERE t.task_id = ?
        ");
        $taskStmt->execute([$taskId]);
        $task = $taskStmt->fetch();
        
        if (!$task) return [];
        
        $userIds = [];
        
        // Agregar creador de la tarea
        if ($task['created_by_user_id']) {
            $userIds[] = (int)$task['created_by_user_id'];
        }
        
        // Agregar usuario asignado directamente
        if ($task['assigned_to_user_id']) {
            $userIds[] = (int)$task['assigned_to_user_id'];
        }
        
        // Agregar usuarios asignados a través de Task_Assignments
        $assignStmt = $db->prepare("SELECT DISTINCT user_id FROM Task_Assignments WHERE task_id = ?");
        $assignStmt->execute([$taskId]);
        $assignments = $assignStmt->fetchAll();
        foreach ($assignments as $a) {
            $userIds[] = (int)$a['user_id'];
        }
        
        // Agregar participantes del proyecto
        $participantStmt = $db->prepare("SELECT DISTINCT user_id FROM Project_Participants WHERE project_id = ?");
        $participantStmt->execute([$task['project_id']]);
        $participants = $participantStmt->fetchAll();
        foreach ($participants as $p) {
            $userIds[] = (int)$p['user_id'];
        }
        
        // Eliminar duplicados y excluir usuario si se especifica
        $userIds = array_unique($userIds);
        if ($excludeUserId) {
            $userIds = array_filter($userIds, function($id) use ($excludeUserId) {
                return (int)$id !== (int)$excludeUserId;
            });
            // Reindexar array después de array_filter
            $userIds = array_values($userIds);
        }
        
        if (empty($userIds)) return [];
        
        // Obtener información de usuarios con email
        $placeholders = implode(',', array_fill(0, count($userIds), '?'));
        $userStmt = $db->prepare("
            SELECT user_id, full_name, email 
            FROM Users 
            WHERE user_id IN ($placeholders) 
            AND email IS NOT NULL 
            AND email != '' 
            AND is_active = 1
        ");
        $userStmt->execute($userIds);
        return $userStmt->fetchAll();
    }
    
    /**
     * Obtener todos los usuarios involucrados en una subtarea
     */
    private function getSubtaskInvolvedUsers($subtaskId, $excludeUserId = null) {
        $db = Database::getConnection();
        
        // Obtener información de la subtarea y su tarea padre
        $subtaskStmt = $db->prepare("
            SELECT s.subtask_id, s.title, s.created_by_user_id, s.assigned_to_user_id, s.task_id,
                   t.task_name, t.project_id, t.created_by_user_id as task_created_by,
                   p.project_name, p.clan_id
            FROM Subtasks s
            JOIN Tasks t ON s.task_id = t.task_id
            JOIN Projects p ON t.project_id = p.project_id
            WHERE s.subtask_id = ?
        ");
        $subtaskStmt->execute([$subtaskId]);
        $subtask = $subtaskStmt->fetch();
        
        if (!$subtask) return [];
        
        $userIds = [];
        
        // Agregar creador de la subtarea
        if ($subtask['created_by_user_id']) {
            $userIds[] = (int)$subtask['created_by_user_id'];
        }
        
        // Agregar usuario asignado directamente a la subtarea
        if ($subtask['assigned_to_user_id']) {
            $userIds[] = (int)$subtask['assigned_to_user_id'];
        }
        
        // Agregar usuarios asignados a través de Subtask_Assignments
        $assignStmt = $db->prepare("SELECT DISTINCT user_id FROM Subtask_Assignments WHERE subtask_id = ?");
        $assignStmt->execute([$subtaskId]);
        $assignments = $assignStmt->fetchAll();
        foreach ($assignments as $a) {
            $userIds[] = (int)$a['user_id'];
        }
        
        // Agregar creador de la tarea padre
        if ($subtask['task_created_by']) {
            $userIds[] = (int)$subtask['task_created_by'];
        }
        
        // Agregar usuarios asignados a la tarea padre
        $taskAssignStmt = $db->prepare("
            SELECT DISTINCT user_id 
            FROM Task_Assignments 
            WHERE task_id = ?
            UNION
            SELECT assigned_to_user_id as user_id
            FROM Tasks
            WHERE task_id = ? AND assigned_to_user_id IS NOT NULL
        ");
        $taskAssignStmt->execute([$subtask['task_id'], $subtask['task_id']]);
        $taskAssignments = $taskAssignStmt->fetchAll();
        foreach ($taskAssignments as $ta) {
            if ($ta['user_id']) {
                $userIds[] = (int)$ta['user_id'];
            }
        }
        
        // Agregar participantes del proyecto
        $participantStmt = $db->prepare("SELECT DISTINCT user_id FROM Project_Participants WHERE project_id = ?");
        $participantStmt->execute([$subtask['project_id']]);
        $participants = $participantStmt->fetchAll();
        foreach ($participants as $p) {
            $userIds[] = (int)$p['user_id'];
        }
        
        // Eliminar duplicados y excluir usuario si se especifica
        $userIds = array_unique($userIds);
        if ($excludeUserId) {
            $userIds = array_filter($userIds, function($id) use ($excludeUserId) {
                return (int)$id !== (int)$excludeUserId;
            });
            // Reindexar array después de array_filter
            $userIds = array_values($userIds);
        }
        
        if (empty($userIds)) return [];
        
        // Obtener información de usuarios con email
        $placeholders = implode(',', array_fill(0, count($userIds), '?'));
        $userStmt = $db->prepare("
            SELECT user_id, full_name, email 
            FROM Users 
            WHERE user_id IN ($placeholders) 
            AND email IS NOT NULL 
            AND email != '' 
            AND is_active = 1
        ");
        $userStmt->execute($userIds);
        return $userStmt->fetchAll();
    }
    
    /**
     * Notificar cuando se agrega un comentario a una tarea
     */
    public function notifyTaskComment($taskId, $commentText, $commentAuthorId, $commentAuthorName) {
        error_log("=== notifyTaskComment INICIO ===");
        error_log("taskId: $taskId, authorId: $commentAuthorId, authorName: $commentAuthorName");
        
        $setting = Notification::getSetting('task_comment');
        error_log("Setting task_comment: " . json_encode($setting));
        
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación task_comment deshabilitada");
            return 0;
        }
        
        $db = Database::getConnection();
        $taskStmt = $db->prepare("
            SELECT t.task_id, t.task_name, t.project_id, p.project_name 
            FROM Tasks t 
            JOIN Projects p ON t.project_id = p.project_id 
            WHERE t.task_id = ?
        ");
        $taskStmt->execute([$taskId]);
        $taskInfo = $taskStmt->fetch();
        if (!$taskInfo) {
            error_log("Tarea no encontrada: $taskId");
            return 0;
        }
        
        $users = $this->getTaskInvolvedUsers($taskId, $commentAuthorId);
        error_log("Usuarios involucrados encontrados: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nuevo comentario en tarea',
                '<p>' . htmlspecialchars($commentAuthorName, ENT_QUOTES, 'UTF-8') . ' agregó un comentario 💬.</p>',
                [
                    ['label' => 'Tarea', 'value' => $taskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $taskInfo['project_name']],
                    ['label' => 'Comentario', 'value' => mb_substr(strip_tags($commentText), 0, 100) . (mb_strlen(strip_tags($commentText)) > 100 ? '...' : '')]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $taskId],
                ['emoji' => '💬']
            );
            
            if ($this->mailer->sendHtml($to, 'Polaris • Nuevo comentario en tarea', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('task_comment mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifyTaskComment FIN - Enviados: $count ===");
        return $count;
    }
    
    /**
     * Notificar cuando se agrega un adjunto a una tarea
     */
    public function notifyTaskAttachment($taskId, $fileName, $attachmentAuthorId, $attachmentAuthorName) {
        error_log("=== notifyTaskAttachment INICIO ===");
        error_log("taskId: $taskId, fileName: $fileName, authorId: $attachmentAuthorId");
        
        $setting = Notification::getSetting('task_attachment');
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación task_attachment deshabilitada");
            return 0;
        }
        
        $db = Database::getConnection();
        $taskStmt = $db->prepare("
            SELECT t.task_id, t.task_name, t.project_id, p.project_name 
            FROM Tasks t 
            JOIN Projects p ON t.project_id = p.project_id 
            WHERE t.task_id = ?
        ");
        $taskStmt->execute([$taskId]);
        $taskInfo = $taskStmt->fetch();
        if (!$taskInfo) return 0;
        
        $users = $this->getTaskInvolvedUsers($taskId, $attachmentAuthorId);
        error_log("Usuarios involucrados encontrados: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nuevo adjunto en tarea',
                '<p>' . htmlspecialchars($attachmentAuthorName, ENT_QUOTES, 'UTF-8') . ' agregó un archivo adjunto 📎.</p>',
                [
                    ['label' => 'Tarea', 'value' => $taskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $taskInfo['project_name']],
                    ['label' => 'Archivo', 'value' => htmlspecialchars($fileName, ENT_QUOTES, 'UTF-8')]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $taskId],
                ['emoji' => '📎']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Nuevo adjunto en tarea', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('task_attachment mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifyTaskAttachment FIN - Enviados: $count ===");
        return $count;
    }
    
    /**
     * Notificar cuando se crea una subtarea
     */
    public function notifySubtaskCreated($subtaskId, $subtaskTitle, $taskId, $createdByUserId, $createdByName) {
        error_log("=== notifySubtaskCreated INICIO ===");
        error_log("subtaskId: $subtaskId, taskId: $taskId, createdBy: $createdByUserId");
        
        $setting = Notification::getSetting('subtask_created');
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación subtask_created deshabilitada");
            return 0;
        }
        
        $db = Database::getConnection();
        $taskStmt = $db->prepare("
            SELECT t.task_id, t.task_name, t.project_id, p.project_name 
            FROM Tasks t 
            JOIN Projects p ON t.project_id = p.project_id 
            WHERE t.task_id = ?
        ");
        $taskStmt->execute([$taskId]);
        $taskInfo = $taskStmt->fetch();
        if (!$taskInfo) {
            error_log("Tarea no encontrada: $taskId");
            return 0;
        }
        
        $users = $this->getTaskInvolvedUsers($taskId, $createdByUserId);
        error_log("Usuarios involucrados encontrados: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nueva subtarea creada',
                '<p>' . htmlspecialchars($createdByName, ENT_QUOTES, 'UTF-8') . ' creó una nueva subtarea 📋.</p>',
                [
                    ['label' => 'Subtarea', 'value' => htmlspecialchars($subtaskTitle, ENT_QUOTES, 'UTF-8')],
                    ['label' => 'Tarea', 'value' => $taskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $taskInfo['project_name']]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $taskId],
                ['emoji' => '📋']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Nueva subtarea creada', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('subtask_created mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifySubtaskCreated FIN - Enviados: $count ===");
        return $count;
    }
    
    /**
     * Notificar cuando se agrega un comentario a una subtarea
     */
    public function notifySubtaskComment($subtaskId, $commentText, $commentAuthorId, $commentAuthorName) {
        error_log("=== notifySubtaskComment INICIO ===");
        error_log("subtaskId: $subtaskId, authorId: $commentAuthorId");
        
        $setting = Notification::getSetting('subtask_comment');
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación subtask_comment deshabilitada");
            return 0;
        }
        
        $db = Database::getConnection();
        $subtaskStmt = $db->prepare("
            SELECT s.subtask_id, s.title, s.task_id, t.task_name, t.project_id, p.project_name
            FROM Subtasks s
            JOIN Tasks t ON s.task_id = t.task_id
            JOIN Projects p ON t.project_id = p.project_id
            WHERE s.subtask_id = ?
        ");
        $subtaskStmt->execute([$subtaskId]);
        $subtaskInfo = $subtaskStmt->fetch();
        if (!$subtaskInfo) {
            error_log("Subtarea no encontrada: $subtaskId");
            return 0;
        }
        
        $users = $this->getSubtaskInvolvedUsers($subtaskId, $commentAuthorId);
        error_log("Usuarios involucrados encontrados: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nuevo comentario en subtarea',
                '<p>' . htmlspecialchars($commentAuthorName, ENT_QUOTES, 'UTF-8') . ' agregó un comentario 💬.</p>',
                [
                    ['label' => 'Subtarea', 'value' => htmlspecialchars($subtaskInfo['title'], ENT_QUOTES, 'UTF-8')],
                    ['label' => 'Tarea', 'value' => $subtaskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $subtaskInfo['project_name']],
                    ['label' => 'Comentario', 'value' => mb_substr(strip_tags($commentText), 0, 100) . (mb_strlen(strip_tags($commentText)) > 100 ? '...' : '')]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $subtaskInfo['task_id']],
                ['emoji' => '💬']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Nuevo comentario en subtarea', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('subtask_comment mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifySubtaskComment FIN - Enviados: $count ===");
        return $count;
    }
    
    /**
     * Notificar cuando se agrega un adjunto a una subtarea
     */
    public function notifySubtaskAttachment($subtaskId, $fileName, $attachmentAuthorId, $attachmentAuthorName) {
        error_log("=== notifySubtaskAttachment INICIO ===");
        error_log("subtaskId: $subtaskId, fileName: $fileName, authorId: $attachmentAuthorId");
        
        $setting = Notification::getSetting('subtask_attachment');
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación subtask_attachment deshabilitada");
            return 0;
        }
        
        $db = Database::getConnection();
        $subtaskStmt = $db->prepare("
            SELECT s.subtask_id, s.title, s.task_id, t.task_name, t.project_id, p.project_name
            FROM Subtasks s
            JOIN Tasks t ON s.task_id = t.task_id
            JOIN Projects p ON t.project_id = p.project_id
            WHERE s.subtask_id = ?
        ");
        $subtaskStmt->execute([$subtaskId]);
        $subtaskInfo = $subtaskStmt->fetch();
        if (!$subtaskInfo) {
            error_log("Subtarea no encontrada: $subtaskId");
            return 0;
        }
        
        $users = $this->getSubtaskInvolvedUsers($subtaskId, $attachmentAuthorId);
        error_log("Usuarios involucrados encontrados: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nuevo adjunto en subtarea',
                '<p>' . htmlspecialchars($attachmentAuthorName, ENT_QUOTES, 'UTF-8') . ' agregó un archivo adjunto 📎.</p>',
                [
                    ['label' => 'Subtarea', 'value' => htmlspecialchars($subtaskInfo['title'], ENT_QUOTES, 'UTF-8')],
                    ['label' => 'Tarea', 'value' => $subtaskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $subtaskInfo['project_name']],
                    ['label' => 'Archivo', 'value' => htmlspecialchars($fileName, ENT_QUOTES, 'UTF-8')]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $subtaskInfo['task_id']],
                ['emoji' => '📎']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Nuevo adjunto en subtarea', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('subtask_attachment mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifySubtaskAttachment FIN - Enviados: $count ===");
        return $count;
    }
    
    /**
     * Notificar cuando se asignan usuarios a una subtarea
     */
    public function notifySubtaskAssigned($subtaskId, $assignedUserIds) {
        error_log("=== notifySubtaskAssigned INICIO ===");
        error_log("subtaskId: $subtaskId, assignedUserIds: " . json_encode($assignedUserIds));
        
        $setting = Notification::getSetting('subtask_assigned');
        if (!(int)($setting['is_enabled'] ?? 1)) {
            error_log("Notificación subtask_assigned deshabilitada");
            return 0;
        }
        
        if (empty($assignedUserIds)) {
            error_log("No hay usuarios asignados");
            return 0;
        }
        
        $assignedUserIds = array_values(array_unique(array_map('intval', $assignedUserIds)));
        $db = Database::getConnection();
        
        $subtaskStmt = $db->prepare("
            SELECT s.subtask_id, s.title, s.task_id, t.task_name, t.project_id, p.project_name
            FROM Subtasks s
            JOIN Tasks t ON s.task_id = t.task_id
            JOIN Projects p ON t.project_id = p.project_id
            WHERE s.subtask_id = ?
        ");
        $subtaskStmt->execute([$subtaskId]);
        $subtaskInfo = $subtaskStmt->fetch();
        if (!$subtaskInfo) {
            error_log("Subtarea no encontrada: $subtaskId");
            return 0;
        }
        
        // Obtener usuarios asignados con email
        $placeholders = implode(',', array_fill(0, count($assignedUserIds), '?'));
        $userStmt = $db->prepare("
            SELECT user_id, full_name, email 
            FROM Users 
            WHERE user_id IN ($placeholders) 
            AND email IS NOT NULL 
            AND email != '' 
            AND is_active = 1
        ");
        $userStmt->execute($assignedUserIds);
        $users = $userStmt->fetchAll();
        error_log("Usuarios encontrados con email: " . count($users));
        
        $count = 0;
        $appUrl = defined('APP_URL') ? APP_URL : 'https://rinotrack.rinorisk.com/alfa/public/';
        
        foreach ($users as $user) {
            $to = $user['email'];
            error_log("Enviando correo a: $to");
            
            $html = EmailTemplate::render(
                'Nueva subtarea asignada',
                '<p>Se te ha asignado una nueva subtarea ✅.</p>',
                [
                    ['label' => 'Subtarea', 'value' => htmlspecialchars($subtaskInfo['title'], ENT_QUOTES, 'UTF-8')],
                    ['label' => 'Tarea', 'value' => $subtaskInfo['task_name']],
                    ['label' => 'Proyecto', 'value' => $subtaskInfo['project_name']]
                ],
                ['label' => 'Ver tarea', 'url' => $appUrl . '?route=clan_leader/get-task-details&task_id=' . $subtaskInfo['task_id']],
                ['emoji' => '📋']
            );
            if ($this->mailer->sendHtml($to, 'Polaris • Subtarea asignada', $html)) {
                error_log("Correo enviado exitosamente a: $to");
                $count++;
            } else {
                error_log('subtask_assigned mail fail to ' . $to . ' - ' . $this->mailer->getLastError());
            }
        }
        
        error_log("=== notifySubtaskAssigned FIN - Enviados: $count ===");
        return $count;
    }

    private function parseRecipients($csv) {
        if (!$csv) return [];
        $parts = array_map('trim', explode(',', $csv));
        return array_values(array_filter($parts, function($e){ return filter_var($e, FILTER_VALIDATE_EMAIL); }));
    }
}


